<?php
/**
 * LeadsFlow Pro
 * (c) IMSuccessCenter.com, 2015
 */


class leadflow_mobile_api extends leadflow {
  static $action = '';


  static function get_access_details() {
    $options = get_option(LF_OPTIONS, array());
    $timestamp = get_option('lf_installation_timestamp');
    $out = array('hash' => '', 'redirect_url' => '', 'api_endpoint' => '', 'api_key' => '');

    if (empty($options['api_access']) || empty($options['api_key']) || strlen($options['api_key']) !== 16) {
      return $out;
    }

    $out['hash'] = base64_encode(rtrim(home_url(), '/') . '/|' . $options['api_key']);
    $out['api_endpoint'] = rtrim(home_url(), '/') . '/';
    $out['redirect_url'] = add_query_arg(array('lfp_mobile_install' => substr($timestamp, -6)), $out['api_endpoint']);
    $out['api_key'] = $options['api_key'];

    return $out;
  } // get_access_details


  // add custom query var for API endpoint
  static function add_query_vars($vars) {
    $options = get_option(LF_OPTIONS, array());

    if (!empty($options['api_access']) && !empty($options['api_key']) && strlen($options['api_key']) === 16) {
      $vars[] = 'leadflow_api_action';
      $vars[] = 'leadflow_api_key';
      $vars[] = 'leadflow_api_search';
      $vars[] = 'leadflow_api_offset';
      $vars[] = 'lfp_mobile_install';
    }

    return $vars;
  } // add_query_vars


  static function process() {
    $options = get_option(LF_OPTIONS, array());
    if (empty($options['api_access']) || empty($options['api_key']) || strlen($options['api_key']) != 16) {
      wp_send_json_error('API access is disabled.');
    }

    self::do_redirect();

    self::check_api_key();
    self::check_action();

    self::send_headers();
    switch (self::$action) {
      case 'hello':
        self::do_hello();
      break;
      case 'dashboard':
        self::do_dashboard();
      break;
      case 'signup_activity':
        self::do_signup_activity();
      break;
      case 'optin_pages':
        self::do_optin_pages();
      break;
      case 'ab_tests':
        self::do_ab_tests();
      break;
      case 'lead_flows':
        self::do_lead_flows();
      break;
      case 'subscribers':
        self::do_subscribers();
      break;
      default:
        wp_send_json_error('Unknown error.');
    }
  } // process


  static function send_headers() {
    header('Cache-Control: no-store, no-cache, must-revalidate, max-age=0');
    header('Cache-Control: post-check=0, pre-check=0', false);
    header('Pragma: no-cache');
  } // send_headers


  static function get_required_data() {
    $data = array();
    $data['site_name'] = get_bloginfo('name');
    $data['site_url'] = home_url();
    $data['site_url'] = rtrim($data['site_url']) . '/';

    return $data;
  } // get_required_data


  static function do_signup_activity() {
    $data = self::get_required_data();

    $events = leadflow_stats::events_get_formatted_mobile_activity(150);
    $data['data'] = $events;

    wp_send_json_success($data);
  } // do_signup_activity


  static function do_hello() {
    $data = self::get_required_data();

    if (function_exists('get_site_icon_url') && get_site_icon_url(256)) {
      $data['site_favicon'] = get_site_icon_url(256);
    } else {
      $data['site_favicon'] = LF_PLUGIN_URL . '/images/default-site-icon.png';
    }
    $data['wp_ver'] = get_bloginfo('version');
    $data['lf_ver'] = str_replace('.', '', leadflow::$version);
    if (strlen($data['lf_ver']) == 2) {
      $data['lf_ver'] = $data['lf_ver'] . '0';
    }

    wp_send_json_success($data);
  } // do_hello


  static function do_optin_pages() {
    $data = self::get_required_data();
    $optins = array();

    $tmp_optins = get_posts(array('post_type' => 'optin-pages', 'posts_per_page' => -1, 'order' => 'DESC', 'orderby' => 'ID', 'post_status' => 'publish'));

    foreach ($tmp_optins as $tmp) {
      $optin = array();
      $stats = leadflow_stats::get_optin_stats($tmp->ID, 30);

      $optin['id'] = $tmp->ID;
      $optin['name'] = get_the_title($tmp->ID);
      $optin['type_id'] = leadflow_optin_common::get_optin_type($tmp->ID);
      $optin['type'] = leadflow_optin_type::$types[$optin['type_id']];
      $optin['subscribers'] = $stats['conversions'];
      $optin['views'] = $stats['views'];
      $optin['conversion_rate'] = $stats['conversion_rate'];
      $optin['trend_views'] = $stats['trend_views'];
      $optin['trend_conversions'] = $stats['trend_conversions'];
      $optin['created'] = get_the_date('Y-m-d H:i', $tmp->ID);
      $optin['modified'] = get_the_modified_date('Y-m-d H:i', $tmp->ID);
      $optin['sign_ups_30d'] = $stats['history'];

      $optins[] = $optin;
    }
    $data['data'] = $optins;

    wp_send_json_success($data);
  } // do_optin_pages


  static function do_lead_flows() {
    $data = self::get_required_data();
    $flows = array();

    $tmp_flows = get_posts(array('post_type' => 'lead-flows', 'posts_per_page' => -1, 'order' => 'DESC', 'orderby' => 'ID', 'post_status' => 'publish'));
    foreach ($tmp_flows as $tmp) {
      $flow = array();
      $stats = leadflow_stats::get_flow_stats($tmp->ID, 30);

      $flow['id'] = $tmp->ID;
      $flow['name'] = get_the_title($tmp->ID);
      $flow['type_id'] = leadflow_flow_common::get_flow_type($tmp->ID);
      $flow['type'] = leadflow_flow_type::$types[$flow['type_id']];
      $flow['views'] = $stats['views'];
      $flow['passes'] = $stats['passes'];
      $flow['new_subscribers'] = $stats['new_subscribers'];
      $flow['trend_passes'] = $stats['trend_passes'];
	  $flow['trend_views'] = $stats['trend_views'];
      $flow['trend_new_subscribers'] = $stats['trend_new_subscribers'];
      $flow['created'] = get_the_date('Y-m-d H:i', $tmp->ID);
      $flow['modified'] = get_the_modified_date('Y-m-d H:i', $tmp->ID);
      $flow['actions_30d'] = $stats['history'];
      $flows[] = $flow;
    }
    $data['data'] = $flows;

    wp_send_json_success($data);
  } // do_lead_flows


  static function do_ab_tests() {
    $data = self::get_required_data();
    $tests = array();

    $ab_tests = get_posts(array('post_type' => 'leadflow-abtests', 'posts_per_page' => -1, 'order' => 'DESC', 'orderby' => 'ID', 'post_status' => 'publish'));
    foreach ($ab_tests as $tmp) {
      $test = array();
      $stats = leadflow_stats::get_ab_test_stats($tmp->ID, 30);
      $ab_details = leadflow_ab_test_common::get_details($tmp->ID);

      $test['id'] = $tmp->ID;
      $test['name'] = get_the_title($tmp->ID);
      $test['type_id'] = leadflow_ab_test_common::get_ab_test_type($tmp->ID);
      $test['type'] = leadflow_optin_type::$types[$test['type_id']];
      $test['optins_cnt'] = $ab_details['count'];
      $test['views'] = $stats['views'];
      $test['conversions'] = $stats['conversions'];
      $test['conversion_rate'] = $stats['conversion_rate'];
      $test['trend_views'] = $stats['trend_views'];
      $test['trend_conversions'] = $stats['trend_conversions'];
      $flow['created'] = get_the_date('Y-m-d H:i', $tmp->ID);
      $flow['modified'] = get_the_modified_date('Y-m-d H:i', $tmp->ID);
      $test['actions_30d'] = $stats['history'];
      $tests[] = $test;
    }
    $data['data'] = $tests;

    wp_send_json_success($data);
  } // do_ab_tests


  static function do_subscribers() {
    $offset = abs((int) get_query_var('leadflow_api_offset'));
    $search = trim(get_query_var('leadflow_api_search'));
    $data = self::get_required_data();
    $subs = array();

    $tmp_subs = get_posts(array('post_type' => 'leadflow-subscribers', 'posts_per_page' => 50, 's' => $search, 'offset' => $offset, 'order' => 'DESC', 'orderby' => 'ID', 'post_status' => 'publish'));

    foreach ($tmp_subs as $sub_org) {
      $sub = leadflow_subscribers_common::get_details($sub_org);
      if (!$sub) {
        continue;
      }

      $sub['events'] = leadflow_stats::events_get_formatted_mobile_subscriber($sub['events']);

      $sub['vars'] = array();
      foreach ($sub['custom_vars'] as $key => $value) {
        $sub['vars'][] = array('key' => $key, 'value' => $value);
      } // foreach vars
      unset($sub['custom_vars']);
      $subs[] = $sub;
    } // foreach tmp_sub

    if ($subs) {
      $data['data_empty'] = false;
    } else {
      $data['data_empty'] = true;
    }
    $data['data'] = $subs;
    $data['res_count'] = sizeof($subs);

    wp_send_json_success($data);
  } // do_subscribers


  static function do_dashboard() {
    $data = self::get_required_data();
    $subs_stats = leadflow_stats::get_subscribers_stats(30, 5);

    $data['optins_count'] = (int) @wp_count_posts('optin-pages')->publish;
    $data['ab_tests_count'] = (int) @wp_count_posts('leadflow-abtests')->publish;
    $data['flows_count'] = (int) @wp_count_posts('lead-flows')->publish;
    $data['subscribers_count'] = $subs_stats['total'];

    $data['new_subscribers_24h'] = $subs_stats['new_24h'];
    $data['trend_subscribers_24h'] = $subs_stats['trend_24h'];

    $tmp = leadflow_stats::get_flows_stats();
    $data['flow_passes_24h'] = $tmp['passes_24h'];
    $data['flow_trend_passes_24h'] = $tmp['trend_passes'];

    $tmp = leadflow_stats::get_optins_stats();
    $data['optin_views_24h'] = $tmp['views_24h'];
    $data['optin_trend_views_24h'] = $tmp['trend_views'];

    $data['sign_ups_30d'] = $subs_stats['history'];
     
    $tmp = array();
    foreach ($subs_stats['countries'] as $country => $val) {
      $tmp[$country] = $val['cnt'];
    }
    if (empty($tmp)) {
      $data['subscribers_countries'] = array('__empty' => 1);
    } else {
      $data['subscribers_countries'] = $tmp;
    }

    wp_send_json_success($data);
  } // do_dashboard


  static function check_action() {
    $allowed_actions = array('hello', 'dashboard', 'signup_activity', 'optin_pages', 'ab_tests', 'lead_flows', 'subscribers');
    $action = get_query_var('leadflow_api_action');
    $action = trim(strtolower($action));

    if (!in_array($action, $allowed_actions)) {
      wp_send_json_error('Unknown action.');
    } else {
      self::$action = $action;
    }
  } // check_action


  static function check_api_key() {
    $key = get_query_var('leadflow_api_key');
    $key = trim(strtoupper($key));

    $options = get_option(LF_OPTIONS, array());

    if ($key != $options['api_key']) {
      wp_send_json_error('Invalid API key.');
    }
  } // check_api_key


  static function do_redirect() {
    $details = self::get_access_details();
    $link = 'leadsflowpro://' . $details['hash'];
    $timestamp = get_option('lf_installation_timestamp');

    if (get_query_var('lfp_mobile_install') == substr($timestamp, -6)) {
      self::send_headers();

      leadflow_mobile_app_view::view_mobile_app_install($link);

      die();
    } elseif (get_query_var('lfp_mobile_install')) {
      wp_send_json_error('Invalid APP install key.');
    }
  } // do_redirect
} // leadflow_mobile_api