<?php
/*
Plugin name: LeadsFlow Pro
Version: 1.08
Author: Sean Donahoe
Plugin URI: https://www.leadsflowpro.com/
Author URI: https://www.leadsflowpro.com/
Description: The ultimate lead generation, tracking, integration and management platform for WordPress. Collect leads and channel those leads in to any system with a few clicks of the mouse and finally take control of your leads.
*/

define('LF_PLUGIN_PATH', plugin_dir_path(__FILE__));
define('LF_PLUGIN_URL', plugins_url('', __FILE__));
define('LF_TEMPLATE_PATH', LF_PLUGIN_URL . '/optin_templates');

// Templates
define('LF_TEMPLATES_SERVER', 'https://www.leadsflowpro.com/members/');
define('LF_TEMPLATES_OPTION', 'lf_optin_templates');
define('LF_TEMPLATES_CRON', 'lf_optin_templates_cron');
define('LF_INSTALLED_TEMPLATES', 'lf_installed_templates');

// GeoIP
define('LF_GEOIP_URL', 'https://www.leadsflowpro.com/members/');
define('LF_GEOIP_OPTION', 'lf_geoip');

// LF Contants
define('LF_OPTIONS', 'lf_options');
define('LF_OPTIN_LISTS', 'lf_optin_lists');
define('LF_META', 'lf_meta');
define('LF_EVENTS', 'lf_events');
define('LF_POINTERS', 'lf_pointers');

// register autoloader
if (function_exists('spl_autoload_register')) {
  spl_autoload_register(array('leadflow', 'autoload'), false, false);
} elseif(!function_exists('__autoload')) {
  function __autoload($class) {
    leadflow::autoload($class);
  }
} else {
  global $error_autoloader;
  $error_autoloader = true;
}



global $lfpauthsettings;
$lfpauthsettings=array(
    "headerlogo"=>LF_PLUGIN_URL."/images/header-bar-logo.jpg",
    "headerbg"=>LF_PLUGIN_URL."/images/header-bar.jpg",
    "headerheight"=>"80px",
    "icon" => LF_PLUGIN_URL."/images/menu-logo.png",
    "pluginname" => "LeadsFlow Pro",
    "pluginprefix" => "lfp_",
    "pluginfile" => __FILE__,
    "pluginadmin" => "admin.php?page=leadflow",
    "menutitle" => "Activate<br />LeadsFlow Pro",
    "parentmenu" => 'leadflow',
    "serverurl" => "leadsflowpro.com",
    "authlocation" => "/members/wp-content/plugins/license-checker/authorize_domain.php",
    "memberurl" => "https://leadsflowpro.com/members/wp-login.php?action=lostpassword",
    "minlevel" => 1,
);


global $leadflow_lc;


$leadflow_lc = new leadflow_lc($lfpauthsettings);
if(!$leadflow_lc->CheckLicense()) return;



// main class
class leadflow {
  static $version = '1.08';
  static $templates;

  // autoload class files
  static function autoload($class_name) {
    if (substr($class_name, 0, 9) != 'leadflow_') {
      return;
    }

    $class_name = str_replace('leadflow_', '', $class_name);
    $file_name = $class_name . '.php';

    if (file_exists(LF_PLUGIN_PATH . 'includes/' . $file_name)) {
      require_once LF_PLUGIN_PATH . 'includes/' . $file_name;
      return;
    }
    if (file_exists(LF_PLUGIN_PATH . 'meta_boxes/' . $file_name)) {
      require_once LF_PLUGIN_PATH . 'meta_boxes/' . $file_name;
      return;
    }
    if (file_exists(LF_PLUGIN_PATH . 'views/' . $file_name)) {
      require_once LF_PLUGIN_PATH . 'views/' . $file_name;
      return;
    }
  } // autoload


  // hook everything up
  static function init() {
    global $attached_optins;

    // die before execution
    if (!self::do_mandatory_checks() || !self::do_environment_check()) {
      return false;
    }

    self::check_default_options();
    self::register_table_names();

    leadflow_admin_cpt::register_post_types();
    leadflow_admin_cpt::register_taxonomies();

    self::load_google_fonts();

    add_filter('pre_set_site_transient_update_plugins', array('leadflow_common', 'check_plugin_updates'));
    add_filter('plugins_api', array('leadflow_common', 'plugin_api_call'), 10, 3);

    leadflow_cron::setup();
    leadflow_cron::add_actions();

    if (is_admin()) {
      
      // Generate default icons
      if (!get_option('lf-icons', false)) {
        $site_icon = get_site_icon_url(256);
        leadflow_common::merge_default_icons($site_icon);
      }

      // adjust memory if needed
      add_filter('admin_memory_limit', array(__CLASS__, 'admin_memory_limit'), 10, 1);

      add_filter('plugin_action_links_' . basename(dirname(__FILE__)) . '/' . basename(__FILE__), array(__CLASS__, 'plugin_action_links'));

      // Custom Bulk Action Hook
      add_action('admin_action_lf-push-all', array('leadflow_optin_common', 'bulk_edit_tags'));

      // admin AJAX endpoints
      leadflow_admin_ajax::register_ajax_endpoints();

      // admin Meta Boxes
      self::register_metaboxes();

      // Admin Enqueues
      add_action('admin_enqueue_scripts', array('leadflow_admin_enqueues', 'admin_enqueues'));

      // Optin Save & Preview
      add_action('admin_enqueue_scripts', array('leadflow_optin_preview', 'admin_enqueues'));
      add_action('admin_print_footer_scripts', array('leadflow_optin_preview', 'print_preview_dialog'));

      // Subscribers Dialog - Push through flow and Export
      add_action('admin_print_footer_scripts', array('leadflow_bulk', 'print_push_subscribers_dialog'));
      add_action('admin_print_footer_scripts', array('leadflow_bulk', 'print_export_subscribers_dialog'));

      // Save Optin, Leadflow, Subscribers
      self::admin_save_actions();

      // Various update messages
      leadflow_admin_cpt::update_messages();

      // Admin Menu Customization
      add_action('admin_menu', array('leadflow', 'add_options_page'));
      add_filter('custom_menu_order', array('leadflow', 'custom_menu_order'));

      // Table Columns Alterations
      self::admin_html_table_columns_alterations();

      // Upload filter
      add_filter('upload_mimes', array('leadflow', 'geoip_dat_filter'));

      // Subscribers Search Filters
      add_action('admin_print_footer_scripts', array('leadflow_optin_admin_common', 'subscribers_search_filters'));

      // Subscriber Sortable Columns
      add_filter('manage_edit-leadflow-subscribers_sortable_columns', array('leadflow', 'subscribers_sortable'));
      add_action('pre_get_posts', array('leadflow', 'subscribers_sortable_pre_get_posts'));

      add_action('delete_post', array('leadflow_stats', 'delete_stats'));
      add_action('admin_enqueue_scripts', array('leadflow_pointers', 'enqueue_files'));
      add_filter('heartbeat_received', array('leadflow_dashboard', 'heartbeat_received'), 10, 2 );
      add_action('admin_enqueue_scripts', array('leadflow_dashboard', 'setup_js_vars'));
      // On Delete Subscriber
      add_action('transition_post_status', array('leadflow_subscribers_common', 'skip_trash'), 10, 3);

      // Force list VIEW mode
      add_action('load-edit.php', array('leadflow', 'force_table_list_view'));

      // update mobile icons when favicon changes
      add_filter('pre_update_option_site_icon', array('leadflow_common', 'on_update_siteicon'), 10, 2);
    } else { // !is_admin()
      add_action('wp_footer', array('leadflow_flow_common', 'enqueue_form_catcher_js'), 10, 1);

      // wp widget shortcode
      add_filter('widget_text', 'do_shortcode');

      // content locker
      add_action('the_content', array('leadflow_page_lock', 'the_content'), 1, 9999);

      // custom optin URL/template logic
      add_action('template_redirect', array('leadflow_optin_conditionals', 'setup_headers'), 1);

      add_action('send_headers', array('leadflow_common', 'send_headers'), 1);
      add_action('template_redirect', array('leadflow_optin_common', 'template_redirect'), 1);

      add_action('wp_print_footer_scripts', array('leadflow_optin_conditionals', 'output_optins_to_footer'));
    }

    // register shortcodes
    self::register_shortcodes();

    leadflow_frontend_ajax::register_ajax_endpoints();
    
	add_action( 'edit_form_after_title', array('leadflow_common', 'check_custom_urls_in_editor') );
    add_filter('query_vars', array('leadflow_mobile_api', 'add_query_vars'));
	add_filter('upload_mimes', array('leadflow','allow_csv_uploads'));
  } // init

  
  static function allow_csv_uploads($mime_types){
		$mime_types['csv'] = 'text/csv';
		return $mime_types;
  }
  

  static function force_table_list_view() {
    $screen = get_current_screen();

    if ($screen->id == 'edit-leadflow-subscribers' || $screen->id == 'edit-lead-flows'
      || $screen->id == 'edit-optin-pages' || $screen->id == 'edit-leadflow-abtests') {
      $_REQUEST['mode'] = 'list';
    }
  } // force_table_list_view


  static function subscribers_sortable_pre_get_posts($query) {
    if(!is_admin()) {
      return;
    }

    $orderby = $query->get( 'orderby');

    if('stars' == $orderby) {
      $query->set('meta_key','stars');
      $query->set('orderby','meta_value_num');
    }
  } // subscribers_sortable


  static function subscribers_sortable($columns) {
    $columns['stars'] = 'stars';
    return $columns;
  } // subscribers_sortable


  // Admin Table Columns Alterations
  static function admin_html_table_columns_alterations() {
    // Various Table Alterations
    add_filter('post_row_actions', array('leadflow', 'post_row_actions'), 1000, 2);
    add_action('admin_action_optin_clone', array('leadflow', 'duplicate_post'));
    #add_action('admin_action_optin_ab_tests', array('leadflow_ab_test_common', 'process_ab_actions'));
    add_action('admin_action_optin_refresh_lists', array('leadflow_form_box', 'refresh_lists'));
    add_filter('posts_where', array('leadflow_optin_common', 'filter_optins_by_id'), 10, 2);

    // Manage optin columns
    add_filter('manage_leadflow-abtests_posts_columns', array('leadflow', 'manage_columns_abtests'), 999, 1);
    add_action('manage_leadflow-abtests_posts_custom_column' , array('leadflow', 'manage_column_abtests'), 10, 2);
    add_filter('manage_optin-pages_posts_columns', array('leadflow', 'manage_columns_optin'), 999, 1);
    add_action('manage_optin-pages_posts_custom_column' , array('leadflow', 'manage_column_optin'), 10, 2);
    add_filter('manage_lead-flows_posts_columns', array('leadflow', 'manage_columns_flow'), 999, 1);
    add_action('manage_lead-flows_posts_custom_column' , array('leadflow', 'manage_column_flow'), 10, 2);

    // Manage subscriber columns
    add_filter('manage_leadflow-subscribers_posts_columns', array('leadflow', 'manage_columns_subscribers'), 999, 1);
    add_action('manage_leadflow-subscribers_posts_custom_column' , array('leadflow', 'manage_column_subscribers'), 10, 2);

    // Manage taxonomy terms table
    add_filter('manage_edit-leadflow-subscriber-tags_columns', array('leadflow', 'manage_terms_columns'), 999, 1);
    add_filter('manage_leadflow-subscriber-tags_custom_column', array('leadflow', 'manage_terms_columns_content'), 999, 3);

    //
    add_action('post_submitbox_start', array('leadflow', 'post_submitbox'));
  } // admin_html_table_columns_alterations


  static function manage_terms_columns_content($deprecated,$column_name,$term_id) {
    switch ($column_name) {
      case 'subscribers':
        $count = get_term($term_id, 'leadflow-subscriber-tags');
        echo '<a href="' . admin_url('edit.php?post_type=leadflow-subscribers&leadflow-subscriber-tags=' . $count->slug . '') . '">' . $count->count . '</a>';
        break;
    }
  } // manage_terms_columns_content


  static function manage_terms_columns($columns) {
    unset($columns['posts']);
    $columns['subscribers'] = 'Subscribers';
    return $columns;
  } // manage_terms_columns


  // All Admin Save Actions
  static function admin_save_actions() {
    add_action('save_post', array('leadflow_page_lock', 'save_post'), 10, 3);
    add_action('save_post_leadflow-abtests', array('leadflow', 'save_ab_test'), 10, 3);
    add_action('save_post_optin-pages', array('leadflow', 'save_post_optin'), 10, 3);
    add_action('save_post_lead-flows', array('leadflow', 'save_post_flow'), 10, 3);
    add_action('save_post_leadflow-subscribers', array('leadflow', 'save_post_subscribers'), 10, 3);

    // taxonomy meta
    add_action('leadflow-subscriber-tags_add_form_fields', array('leadflow', 'subscriber_tags_meta'), 10, 2);
    add_action('leadflow-subscriber-tags_edit_form_fields', array('leadflow', 'edit_subscriber_tags_meta'), 10,2);
    add_action('edited_leadflow-subscriber-tags', array('leadflow', 'save_subscriber_tags_meta'), 10, 2);
    add_action('create_leadflow-subscriber-tags', array('leadflow', 'save_subscriber_tags_meta'), 10, 2);
	
	add_filter('post_row_actions',array('leadflow', 'leadflow_subscribers_row_actions'), 10, 2);
  } // admin_save_actions
	
	
  static function leadflow_subscribers_row_actions($actions, $post){
	  //check for your post type
	  if ($post->post_type =="leadflow-subscribers"){
		  foreach($actions as $action=>$url){
			if($action != 'edit' && $action != 'trash'){
				unset($actions[$action]);	
			}
		  }
	  } else if ($post->post_type =="optin-pages"){
		  foreach($actions as $action=>$url){
			if($action != 'edit' && $action != 'trash'){
				unset($actions[$action]);	
			}
		  }
	  }
	  return $actions;
  }

  // register metaboxes
  static function register_metaboxes() {
    add_action('add_meta_boxes_optin-pages', array('leadflow', 'meta_boxes_optin'), 999, 2);
    add_action('add_meta_boxes_lead-flows', array('leadflow', 'meta_boxes_flow'), 999, 2);
    add_action('add_meta_boxes_leadflow-subscribers', array('leadflow', 'meta_boxes_subscibers'), 999, 2);
    add_action('add_meta_boxes_leadflow-abtests', array('leadflow', 'meta_boxes_abtests'), 999, 2);
    add_action('add_meta_boxes', array('leadflow', 'meta_boxes'), 10, 2);
  } // register_metaboxes


  // register shortcodes
  static function register_shortcodes() {
    $registered_sc = true;

    if (!shortcode_exists('optin-shortcode')) {
      add_shortcode('optin-shortcode', array('leadflow_optin_common', 'shortcode_optin'));
    } else {
      $registered_sc = false;
    }

    if (!shortcode_exists('optin_shortcode')) {
      add_shortcode('optin_shortcode', array('leadflow_optin_common', 'shortcode_optin'));
    } else {
      $registered_sc = false;
    }
	
	if (!shortcode_exists('optin_ab_shortcode')) {
      add_shortcode('optin_ab_shortcode', array('leadflow_ab_test_common', 'shortcode_optin_ab'));
    } else {
      $registered_sc = false;
    }

    if (!shortcode_exists('optin-popup')) {
      add_shortcode('optin-popup', array('leadflow_optin_common', 'shortcode_optin_popup'));
    } else {
      $registered_sc = false;
    }

    if (!shortcode_exists('optin_popup')) {
      add_shortcode('optin_popup', array('leadflow_optin_common', 'shortcode_optin_popup'));
    } else {
      $registered_sc = false;
    }

    if (!shortcode_exists('optin_test_popup')) {
      add_shortcode('optin_test_popup', array('leadflow_optin_common', 'shortcode_optin_test_popup'));
    } else {
      $registered_sc = false;
    }

    if (!shortcode_exists('optin-test-popup')) {
      add_shortcode('optin-test-popup', array('leadflow_optin_common', 'shortcode_optin_test_popup'));
    } else {
      $registered_sc = false;
    }

    if (!$registered_sc) {
      add_action('admin_notices', array(__CLASS__, 'error_shortcode_registration'));
    }
  } // register_shortcodes


  // mandatory checks
  static function do_mandatory_checks() {
    global $error_autoloader;

    if (!version_compare(PHP_VERSION, '5.4',  '>=')) {
      add_action('admin_notices', array(__CLASS__, 'error_min_version_php'));
      return false;
    }

    if (!version_compare(get_bloginfo('version'), '4.4',  '>=')) {
      add_action('admin_notices', array(__CLASS__, 'error_min_version_wp'));
      return false;
    }

    if (!function_exists('gzcompress')) {
      add_action('admin_notices', array(__CLASS__, 'error_gzcompress_required'));
      return false;
    }

    if (!function_exists('base64_encode')) {
      add_action('admin_notices', array(__CLASS__, 'error_base64_encode_required'));
      return false;
    }

    if ($error_autoloader === false) {
      add_action('admin_notices', array(__CLASS__, 'error_no_autoloader'));
      return false;
    }

    return true;
  } // do_mandatory_checks


  // environment check
  static function do_environment_check() {

    $settings = get_option(LF_OPTIONS, array());
    // treba 2x - ostavi! DOLJE!
    if (isset($_POST['lf_settings']['optin_preview_page']) &&
      $_POST['lf_settings']['optin_preview_page'] == '0') {
      add_action('admin_notices', array(__CLASS__, 'error_no_preview_page'));
    }


    if (!isset($_POST['lf_settings']['optin_preview_page']) && ( empty($settings['optin_preview_page']) || 0 === $settings['optin_preview_page']) ) {
      add_action('admin_notices', array(__CLASS__, 'error_no_preview_page'));
    }
    // // treba 2x - ostavi! GoRE!!

    if (!get_option('permalink_structure')) {
      add_action('admin_notices', array(__CLASS__, 'error_no_permalinks'));
    }

    if (!leadflow_geoip::get_geoip_file_status()) {
      add_action('admin_notices', array(__CLASS__, 'error_download_geoip'));
    }

    if (get_transient('lf-geoip-file-download')) {
      delete_transient('lf-geoip-file-download');
      add_action('admin_notices', array(__CLASS__, 'success_download_geoip'));
    }

    if (!@session_start() || !session_id()) {
      add_action('admin_notices', array(__CLASS__, 'error_session_start'));
    }

    return true;
  } // do_environment_check


  // set default options
  static function check_default_options() {
    $options = get_option(LF_OPTIONS, array());
    if (sizeof($options) < leadflow_settings::$defaults) {
      update_option(LF_OPTIONS, $options, true);
    }
  } // check_options


  // admin memory limit
  static function admin_memory_limit($old_value) {
    $options = get_option(LF_OPTIONS, array());
    if (!$options['memory_limit']) {
      return $old_value;
    }

    return $options['memory_limit'];
  } // admin_memory_limit


  // add geoip.dat to allowed filter for upload
  static function geoip_dat_filter($existing_mimes=array()){
    $existing_mimes['dat'] = 'application/octet-stream';
    return $existing_mimes;
  } // geoip_dat_filter


  // add settings link to plugins page
  static function plugin_action_links($links) {
    $settings_link = '<a href="' . admin_url('edit.php?post_type=optin-pages') . '" title="Manage LeadsFlow Pro">Manage LeadsFlow Pro</a>';
    array_unshift($links, $settings_link);

    return $links;
  } // plugin_action_links


  // register custom tables in wpdb
  static function register_table_names() {
    global $wpdb;

    $wpdb->lf_stats = $wpdb->prefix . 'lf_stats';
  } // register_tables


  // success - geo ip loaded
  static function success_download_geoip() {
    echo '<div class="updated"><p>LeadsFlow Pro <b>Geo IP database</b> has been downloaded successfully.</p></div>';
  } // success_download_geoip

  // warning - geo ip not loaded
  static function error_download_geoip() {
    echo '<div class="error"><p>LeadsFlow Pro <b>requires the GeoIP database</b>. <a onclick="location.reload();" href="' . admin_url('admin.php?page=leadflow-settings#3') . '">Please download it</a>.</p></div>';
  } // error_download_geoip


  // warning - unable to hook autoloader
  static function error_no_autoloader() {
    echo '<div class="error"><p>LeadsFlow Pro <b>requires PHP version 5.4</b> or higher and <b>SPL autoloader</b> function to function properly. You\'re using PHP version ' . PHP_VERSION . '. Please contact your hosting company to have your server updated.</p></div>';
  } // error_no_autoloader


  // warning - unable to hook autoloader
  static function error_session_start() {
    echo '<div class="error"><p>LeadsFlow Pro was <b>unable to start a PHP session</b>. Some parts of the plugin will not function properly. Please contact your hosting provider to correct this issue.</p></div>';
  } // error_session_start


  // display warning if shortcode is already registered/in use
  static function error_shortcode_registration() {
    echo '<div class="error"><p>LeadsFlow Pro <b>detected</b> some of shortcodes are already in use by other plugin or by the currently active theme and they will not function properly because of that.</p></div>';
  } // error_shortcode_registration


  // display warning if WP is outdated
  static function error_min_version_wp() {
    echo '<div class="error"><p>LeadsFlow Pro <b>requires WordPress version 4.3</b> or higher to function properly. You\'re using WordPress version ' . get_bloginfo('version') . '. Please <a href="' . admin_url('update-core.php') . '" title="Update WP core">update</a>.</p></div>';
  } // error_min_version_wp


  // display warning if PHP is outdated
  static function error_gzcompress_required() {
    echo '<div class="error"><p>LeadsFlow Pro <b>requires GZCompress enabled</b> to function properly. Please <a href="http://php.net/manual/en/function.gzcompress.php" title="" target="_blank">GZCompress</a>.</p></div>';
  } // error_gzcompress_required


  // display warning if PHP is outdated
  static function error_base64_encode_required() {
    echo '<div class="error"><p>LeadsFlow Pro <b>requires base64 encoding enabled</b> to function properly. Please visit <a href="http://php.net/manual/en/function.base64-encode.php" title="" target="_blank">Base64</a>.</p></div>';
  } // error_base64_encode_required


  // display warning if PHP is outdated
  static function error_min_version_PHP() {
    echo '<div class="error"><p>LeadsFlow Pro <b>requires PHP version 5.4</b> or higher to function properly. You\'re using PHP version ' . PHP_VERSION . '. Please contact your hosting company to have your server updated.</p></div>';
  } // error_min_version_php


  // permalinks are disabled
  static function error_no_permalinks() {
    echo '<div class="error"><p>LeadsFlow Pro <b>requires permalinks</b> to be enabled to function properly. Please <a href="' . admin_url('options-permalink.php') . '" title="Update WP core">enable permalinks</a>.</p></div>';
  } // error_no_permalinks


  // preview page not set
  static function error_no_preview_page() {
    echo '<div class="error"><p>LeadsFlow Pro <b>requires a preview page</b> to be set in order to use the OptIn\'s Preview function <a href="' . admin_url('admin.php?page=leadflow-settings#0') . '" onclick="location.reload();">Set the preview page here</a>.</p></div>';
  } // error_no_preview_page


  // modify action links below cpt in table view
  static function post_row_actions($actions, $post) {
    if ($post->post_type == 'leadflow-subscribers') {
      unset($actions['inline hide-if-no-js']);
      $actions['trash'] = str_replace('Trash', 'Delete', $actions['trash']);
      $actions['edit'] = str_replace('Edit</a>', 'View Details</a>', $actions['edit']);
      return $actions;
    } elseif ($post->post_type == 'optin-pages') {
      unset($actions['inline hide-if-no-js']);
      if (isset($actions['view'])) {
        if (leadflow_optin_common::get_optin_type($post->ID) == 'standalone') {
          $actions['view'] = str_replace('>', ' target="_blank">', $actions['view']);
        } else {
          unset($actions['view']);
        }
      }
      $actions['optin_pages_clone'] = '<a href="' . admin_url('admin.php?action=optin_clone&amp;post=' . $post->ID) . '">Clone</a>';
      $actions['optin_pages_stats'] = '<a href="' . admin_url('admin.php?page=leadflow-stats&stats-view=optin|' . $post->ID) . '">View Statistics</a>';
      return $actions;
    } elseif ($post->post_type == 'lead-flows') {
      unset($actions['inline hide-if-no-js'], $actions['view']);
      if (isset($actions['livepbedit'])) {
       unset($actions['livepbedit'], $actions['pbclonepost'], $actions['pbdraftpost']); 
      }
      $actions['flows_clone'] = '<a href="' . admin_url('admin.php?action=optin_clone&amp;post=' . $post->ID) . '">Clone</a>';
      $actions['flows_stats'] = '<a href="' . admin_url('admin.php?page=leadflow-stats&stats-view=flow|' . $post->ID) . '">View Statistics</a>';
      return $actions;
    } elseif ($post->post_type == 'leadflow-abtests') {
      unset($actions['inline hide-if-no-js'], $actions['view']);
      if (isset($actions['livepbedit'])) {
       unset($actions['livepbedit'], $actions['pbclonepost'], $actions['pbdraftpost']); 
      }
      $actions['flows_stats'] = '<a href="' . admin_url('admin.php?page=leadflow-stats&stats-view=ab|' . $post->ID) . '">View Statistics</a>';
      return $actions;
    }

    return $actions;
  } // post_row_actions


  // add clone link to submitbox
  static function post_submitbox() {
    global $post;

    if (isset($_GET['post']) && $post->post_type == 'optin-pages') {
      echo '<div><a id="clone-optin" href="' . admin_url('admin.php?action=optin_clone&amp;redirect=1&amp;post=' . $post->ID) . '">Clone OptIn';
      echo '</a></div>';
    } elseif (isset($_GET['post']) && $post->post_type == 'lead-flows') {
      echo '<div><a id="clone-flow" href="' . admin_url('admin.php?action=optin_clone&amp;redirect=1&amp;post=' . $post->ID) . '">Clone Flow';
      echo '</a></div>';
    }
  } // post_submitbox


  // prepare for post cloning
  static function duplicate_post($status = ''){
    if (!(isset($_GET['post']) || isset($_POST['post']) || (isset($_REQUEST['action']) && 'optin_clone' == $_REQUEST['action']))) {
      wp_die('No post to duplicate!');
    }

    $id = (isset($_GET['post']) ? $_GET['post'] : $_POST['post']);
    $post = get_post($id);

    if (isset($post) && $post != null) {
      $new_id = self::duplicate_post_create_duplicate($post);

      if (!empty($_GET['redirect'])) {
        wp_redirect( admin_url( 'post.php?action=edit&post=' . $new_id ) );
      } else {
        wp_redirect( admin_url( 'edit.php?post_type=' . $post->post_type) );
      }
      exit;
    } else {
      wp_die('Copy creation failed, could not find original.');
    }
  } // duplicate post


  // do actual post cloning
  static function duplicate_post_create_duplicate($post, $status = '') {
    if ($post->post_type == 'revision') {
      return;
    }

    $new_post_author = wp_get_current_user();

    $new_post = array(
      'menu_order' => $post->menu_order,
      'comment_status' => $post->comment_status,
      'ping_status' => $post->ping_status,
      'post_author' => $new_post_author->ID,
      'post_content' => $post->post_content,
      'post_excerpt' => $post->post_excerpt,
      'post_mime_type' => $post->post_mime_type,
      'post_parent' => $post->post_parent,
      'post_password' => $post->post_password,
      'post_status' => 'draft',
      'post_title' => 'Clone of ' . $post->post_title,
      'post_type' => $post->post_type,
    );

    $new_post_id = wp_insert_post($new_post);

    $post_meta_keys = get_post_custom_keys($post->ID);
    if (!empty($post_meta_keys)) {
      foreach ($post_meta_keys as $meta_key) {
        $meta_values = get_post_custom_values($meta_key, $post->ID);
        foreach ($meta_values as $meta_value) {
          $meta_value = maybe_unserialize($meta_value);
          add_post_meta($new_post_id, $meta_key, $meta_value);
        }
      }
    }

    return $new_post_id;
  } // duplicate_post_create_duplicate


  // modify columns for table view
  static function manage_columns_optin($columns) {
    $new['cb'] = '<input type="checkbox" />';
    $new['title'] = 'Name';
    $new['type-template'] = 'Type &amp; Template';
    $new['custom_url'] = 'Custom URL';
    $new['abtest'] = 'A/B Test';
    $new['conversion_rate'] = 'Conversion Rate';
    $new['stats'] = 'Quick Stats';
    $new['date'] = 'Date';

    return $new;
  } // manage_columns_optin


  // modfiy columns for subscribers
  static function manage_columns_subscribers($columns) {
    $new['cb'] = '<input type="checkbox" />';
    $new['title'] = 'Subscribers E-mail';
    $new['basic-info'] = 'Basic Info';
    $new['stars'] = 'Lead Rating';
    $new['transactions'] = 'Transactions';
    $new['subscriber-tags'] = 'Tags';
    $new['subscriber-system-tags'] = 'OptIns / Flows';
    $new['created'] = 'Created / Last Action';

    return $new;
  } // manage_columns_subscribers


  static function manage_column_subscribers($column, $post_id) {
    global $wpdb;

    $post = get_post($post_id);
    $url = '?post_type=leadflow-subscribers';

    $subscriber_tags = '&leadflow-subscriber-tags=';
    if (!empty($_GET['leadflow-subscriber-tags'])) {
      $subscriber_tags .= $_GET['leadflow-subscriber-tags'] . '+';
    }

    $subscriber_system_tags = '&leadflow-subscriber-system-tags=';
    if (!empty($_GET['leadflow-subscriber-system-tags'])) {
      $subscriber_system_tags .= $_GET['leadflow-subscriber-system-tags'] . '+';
    }

    switch ($column) {
      case 'transactions':
        $ttc = get_post_meta($post_id, 'ttc', true);
        $ttv = get_post_meta($post_id, 'ttv', true);
        echo '$' . number_format((float)$ttv, 1)  . ' (' . (int) $ttc . ')';
      break;
      case 'basic-info':
        $firstname = get_post_meta($post_id, 'firstname', true);
        $lastname = get_post_meta($post_id, 'lastname', true);

        echo '<a class="basic-info-avatar" href="' . get_edit_post_link($post_id) . '">';
        echo get_avatar($post->post_title, 42);
		echo '</a>';
		
		
		echo '<div class="basic-info-text">';
		
		echo '<a href="' . get_edit_post_link($post_id) . '">';
        if (empty($firstname) && empty($lastname)) {
          echo '<i>Name not set</i>';
        } else {
          echo '<b>' . $firstname . ' ' . $lastname . '</b>';
        }
        echo '</a>';

        $tmp = array();

        $city = get_post_meta($post_id, 'city', true);
        $region = get_post_meta($post_id, 'region', true);
        $country = get_post_meta($post_id, 'country', true);

        if (!empty($city)) {
          $tmp[] = $city;
        }

        if (!empty($region)) {
          $tmp[] = $region;
        }

        if (!empty($country)) {
          $tmp[] = $country;
        }

        echo '<br/>';
        if (empty($tmp)) {
          echo '<i>Unknown location</i>';
        } else {
          echo implode(', ', $tmp);
        }
		
		echo '</div>';
		
        break;
      case 'created':
        $m_time = $post->post_date;
        $time = get_post_time('G', true, $post);
        $time_diff = time() - $time;

        if ($time_diff > 0 && $time_diff < DAY_IN_SECONDS) {
          $h_time = sprintf(__('%s ago'), human_time_diff($time));
        } else {
          $h_time = mysql2date(__('Y/m/d'), $m_time);
        }

        echo $h_time . '<br>';
        $m_time = $post->post_modified;
        $time = get_post_modified_time('G', true, $post);
        $time_diff = time() - $time;

        if ($time_diff > 0 && $time_diff < DAY_IN_SECONDS) {
          $h_time = sprintf(__('%s ago'), human_time_diff($time));
        } else {
          $h_time = mysql2date(__('Y/m/d'), $m_time);
        }

        echo $h_time;
        break;
      case 'stars':
        $stars = get_post_meta($post_id, 'stars', true);
        for ($i = 0; $i < 5; $i++) {
          if ($i < $stars && abs($i - $stars) > 0.5) {
            echo '<span class="dashicons dashicons-star-filled"></span>';
          } elseif ($i > $stars || $stars == 0) {
            echo '<span class="dashicons dashicons-star-empty"></span>';
          } elseif ($stars != 0 && abs($i - $stars) == 0.5) {
            echo '<span class="dashicons dashicons-star-half"></span>';
          } else {
            echo '<span class="dashicons dashicons-star-empty"></span>';
          }
        }
        break;
      case 'subscriber-tags':
        $tags = wp_get_post_terms($post_id, 'leadflow-subscriber-tags');
        if ($tags) {
          echo '<ul class="lf-custom-tags">';
          foreach ($tags as $tag) {
            $style = '';
            $color = leadflow_common::get_subscriber_tags_meta($tag->term_id);

            if ($color) {
              $style = 'style="';
              if (isset($color['tag_color'])) {
                $style .= 'background:' . $color['tag_color'] . ';';
              }

              if (isset($color['font_color'])) {
                $style .= 'color:' . $color['font_color'] . ';';
              }
              $style .= '"';
            }
            echo '<li><a href="' . $url . $subscriber_tags . $tag->slug . '" ' . $style . '>' . $tag->name . '</a></li>';
          } // foreach $tags
          echo '</ul>';
        }
        break;
      case 'subscriber-system-tags':
        $tags = wp_get_post_terms($post_id, 'leadflow-subscriber-system-tags');
        if ($tags) {
          echo '<ul class="lf-custom-tags">';
          foreach ($tags as $tag) {
            $style = '';
            $color = leadflow_common::get_subscriber_tags_meta($tag->term_id);

            // If term is a star - skip it
            if (strpos($tag->slug, 'stars-') !== false) {
              continue;
            }

            if ($color) {
              $style = 'style="';
              if (isset($color['tag_color'])) {
                $style .= 'background:' . $color['tag_color'] . ';';
              }

              if (isset($color['font_color'])) {
                $style .= 'color:' . $color['font_color'] . ';';
              }
              $style .= '"';
            }
            echo '<li><a href="' . $url . $subscriber_system_tags . $tag->slug . '" ' . $style . '>' . $tag->name . '</a></li>';
          } // foreach $tags
          echo '</ul>';
        }
        break;
    } // switch
  } // manage_column_optin


  // modify columns for flow table view
  static function manage_columns_flow($columns) {
    $new['cb'] = '<input type="checkbox" />';
    $new['title'] = 'Name';
    $new['type'] = 'Type';
    $new['custom_url'] = 'Custom URL';
    $new['components'] = '<img src="'.LF_PLUGIN_URL.'/images/components.png" width="30" alt="Components" title="Components" />';
    $new['debugging'] = '<img src="'.LF_PLUGIN_URL.'/images/debugging.png" width="30" alt="Debugging" title="Debugging" />';
    $new['views'] = 'Views';
    $new['actions'] = 'Actions';
    $new['new_subscribers'] = 'New Subscribers';
    $new['date'] = 'Date';

    return $new;
  } // manage_columns_flow



  // get data for custom table columns
  static function manage_column_optin($column, $post_id) {
    global $tmp_optin_table_stats;

    if (empty($tmp_optin_table_stats[$post_id])) {
      $stats = leadflow_stats::get_optin_stats($post_id, 2);
      $tmp_optin_table_stats[$post_id] = $stats;
    } else {
      $stats = $tmp_optin_table_stats[$post_id];
    }

    $meta = get_post_meta($post_id, LF_META, true);

    switch ($column) {
      case 'type-template':
        $type = leadflow_optin_common::get_optin_type($post_id);
        if ($type) {
          echo '<img width="24" src="' . LF_PLUGIN_URL . '/images/optin_types/small/' . $type . '.png" alt="' . leadflow_optin_type::$types[$type] . '" title="'. leadflow_optin_type::$types[$type] . '">&nbsp;&nbsp; ' . leadflow_optin_type::$types[$type];
        } else {
          echo 'Unknown';
        }
        echo '<br>';
        // Get installed templates
        $installed_templates = get_option(LF_INSTALLED_TEMPLATES);
        // Get optin template
        $template = get_post_meta($post_id, 'optin_template', true);
        if (empty($template)) {
          echo '<i>Unknown template</i>';
        } else {
          if (!empty($installed_templates[$template])) {
            echo $installed_templates[$template]['name'];
          } else {
            echo 'Unknown Template';
          }
        }
        break;
      case 'custom_url':
        $customurl = get_post_meta($post_id, 'optin-custom-url', true);
        if ($customurl) {
          echo '<a href="' . home_url() . $customurl . '" target="_blank">' . $customurl . '</a>';
        } else {
          echo '<i>none</i>';
        }
        break;
      case 'abtest':
        $abtest = leadflow_ab_test_common::get_post_ab_test($post_id);
        if (!$abtest) {
          echo '<i>none</i>';
        } else {
          $tmp = leadflow_ab_test_common::get_ab_tests(array('id' => $abtest));
          if ($tmp) {
            echo '<a href="' . admin_url('admin.php?page=leadflow-ab-tests') .'">' . $tmp->post_title . '</a>';
          } else {
            echo '<i>none</i>';
          }
        }
        break;
      case 'views': // not used
        if (empty($stats['views'])) {
          echo '0';
        } else {
          echo number_format($stats['views']) . ' (24h: ' . leadflow_common::trend_number_format($stats['trend_views']) . ')';
        }
        break;
      case 'conversions': // not used
        if (empty($stats['conversions'])) {
          echo '0';
        } else {
          echo number_format($stats['conversions']) . ' (24h: ' . leadflow_common::trend_number_format($stats['trend_conversions']) . ')';
        }
        break;
      case 'conversion_rate':
        if (empty($stats['conversion_rate'])) {
          echo '0%';
        } else {
          echo $stats['conversion_rate'] . '% (24h: ' . leadflow_common::trend_number_format($stats['trend_conversions']) . ')';
        }
        break;
      case 'stats':
        if (empty($meta['second-optin']['disable-second-optin'])) { // 2nd box enabled
          if ($stats['views']) {
            $conv1 = number_format(min(100 * $stats['views_box2'] / $stats['views'], 100), 1);
          } else {
            $conv1 = 0;
          }
          if ($stats['views_box2']) {
            $conv2 = number_format(min(100 * $stats['conversions'] / $stats['views_box2'], 100), 1);
          } else {
            $conv2 = 0;
          }

          echo '<span class="quick-stats-wrapper"><span class="inline-stats">' . number_format($stats['views']) . '<br>views</span><span class="inline-stats-arrow">' . $conv1 .'%</span><span class="inline-stats">' . number_format($stats['views_box2']) . '<br>box #2 views</span><span class="inline-stats-arrow">' . $conv2 . '%</span><span class="inline-stats">' . number_format($stats['conversions']) . '</br>conv</span></span>';
        } else {
          echo '<span class="quick-stats-wrapper"><span class="inline-stats">' . number_format($stats['views']) . '<br>views</span><span class="inline-stats-arrow">' . $stats['conversion_rate'] . '%</span><span class="inline-stats">' . number_format($stats['conversions']) . '</br>conv</span></span>';
        }
        break;
    }
  } // manage_column_optin


  // modify columns for table view
  static function manage_columns_abtests($columns) {
    $new['cb'] = '<input type="checkbox" />';
    $new['title'] = 'Name';
    $new['type'] = 'A/B Type';
    $new['custom_url'] = 'Custom URL';
    $new['opt_count'] = 'Optins Attached';
    $new['views'] = 'Unique Views';
    $new['conversion_rate'] = 'Conversion Rate';
    $new['date'] = 'Date';

    return $new;
  } // manage_columns_optin

  // get data for custom table columns
  static function manage_column_abtests($column, $post_id) {
    global $tmp_optin_table_stats;

    if (empty($tmp_optin_table_stats[$post_id])) {
      $stats = leadflow_stats::get_ab_test_stats($post_id, 2);
      $tmp_optin_table_stats[$post_id] = $stats;
    } else {
      $stats = $tmp_optin_table_stats[$post_id];
    }
	
	
		 
    $meta = get_post_meta($post_id, LF_META, true);

    switch ($column) {
      case 'type':
        $type = leadflow_ab_test_common::get_ab_test_type($post_id);
        if ($type) {
          echo '<img width="24" src="' . LF_PLUGIN_URL . '/images/optin_types/small/' . $type . '.png" alt="' . leadflow_optin_type::$types[$type] . '" title="'. leadflow_optin_type::$types[$type] . '">&nbsp;&nbsp; ' . leadflow_optin_type::$types[$type];
        } else {
          echo 'Unknown';
        }
        break;
      case 'custom_url':
        $customurl = get_post_meta($post_id, 'abtests-custom-url', true);
        if ($customurl) {
          echo '<a href="' . home_url() . $customurl . '" target="_blank">' . $customurl . '</a>';
        } else {
          echo '<i>none</i>';
        }
        break;
      case 'opt_count':
        $count = leadflow_ab_test_common::get_details($post_id);
        if (!empty($count['count'])) {
          echo '<a href="edit.php?post_type=optin-pages&id_filter=' . $count['ids'] . '" target="_blank">' . $count['count'] . '</a>';
        } else {
          echo '<i>0</i>';
        }
        break;
      case 'views': // not used
	    $ab_test_details = leadflow_ab_test_common::get_details($post_id);
		$ab_test_ids=explode(',',$ab_test_details['ids']);
		$optins_stats=array();
		$optin_views=array();
		foreach($ab_test_ids as $optin_id){
		   $optins_stats[$optin_id] = leadflow_stats::get_optin_stats($optin_id, 2);
		   $optin_views[$optin_id]='<a href="'.get_edit_post_link($optin_id).'">'.number_format($optins_stats[$optin_id]['views']).'</a>';
		} 
		
        if (empty($stats['views'])) {
          echo '0';
        } else {
		  echo implode('/',$optin_views);
          echo ' &raquo; <strong>'.number_format($stats['views']) . '</strong> (24h: ' . leadflow_common::trend_number_format($stats['trend_views']) . ')';
        }
        break;
      case 'conversion_rate':
	  	$ab_test_details = leadflow_ab_test_common::get_details($post_id);
		$ab_test_ids=explode(',',$ab_test_details['ids']);
		$optins_stats=array();
		$optin_views=array();
		foreach($ab_test_ids as $optin_id){
		   $optins_stats[$optin_id] = leadflow_stats::get_optin_stats($optin_id, 2);
		   $optin_conversions[$optin_id]='<a href="'.get_edit_post_link($optin_id).'">'.number_format($optins_stats[$optin_id]['conversions']).'</a>';
		} 
		
        if (empty($stats['conversion_rate'])) {
          echo '0%';
        } else {
          echo implode('/',$optin_conversions);
          echo ' &raquo; <strong>'.$stats['conversion_rate'] . '%</strong> (24h: ' . leadflow_common::trend_number_format($stats['trend_conversions']) . ')';
        }
        break;
    }
  } // manage_column_optin


  // get data for custom table flow columns
  static function manage_column_flow($column, $flow_id) {
    global $tmp_flow_table_stats;

    if (empty($tmp_flow_table_stats[$flow_id])) {
      $stats = leadflow_stats::get_flow_stats($flow_id, 2);
      $tmp_flow_table_stats[$flow_id] = $stats;
    } else {
      $stats = $tmp_flow_table_stats[$flow_id];
    }

    switch ($column) {
      case 'type':
        $type = leadflow_flow_common::get_flow_type($flow_id);
        if ($type) {
          echo '<img width="24" src="' . LF_PLUGIN_URL . '/images/flow_types/small/' . $type . '.png" alt="' . leadflow_flow_type::$types[$type] . '" title="'. leadflow_flow_type::$types[$type] . '">&nbsp;&nbsp; ' . leadflow_flow_type::$types[$type];
        } else {
          echo 'Unknown';
        }
        break;
      case 'custom_url':
        if ($tmp = get_post_meta($flow_id, 'flow-custom-url', true)) {
          echo '<a href="' . home_url() . $tmp . '" target="_blank">' . $tmp . '</a>';
        } else {
          echo '<i>none</i>';
        }
        break;
      case 'components':
        echo '<a title="Edit flow components" href="' . admin_url('post.php?action=edit&post=' . $flow_id . '#flow-components') . '">&nbsp;&nbsp;';
        echo sizeof(leadflow_flow_common::get_flow_components($flow_id));
        echo '&nbsp;&nbsp;</a>';
        break;
      case 'debugging':
        $debugging = leadflow_flow_common::get_flow_options('debug', $flow_id);
        echo '<a title="Edit debugging settings" href="' . admin_url('post.php?action=edit&post=' . $flow_id . '#flow-debug') . '">';
        if (!empty($debugging['log'])) {
          echo '<span title="Debugging is enabled" class="dashicons dashicons-yes"></span>';
          if (!empty($debugging['display'])) {
            echo ' <span title="Warning! Debugging output is enabled" class="dashicons dashicons-welcome-view-site"></span>';
          }
        } else {
          echo '<span title="Debugging is disabled" class="dashicons dashicons-no-alt"></span>';
        }
        echo '</a>';
        break;
      case 'views':
        if (empty($stats['views'])) {
          echo '0';
        } else {
          echo number_format($stats['views']) . ' (24h: ' . leadflow_common::trend_number_format($stats['trend_views']) . ')';
        }
        break;
      case 'actions':
        if (empty($stats['passes'])) {
          echo '0';
        } else {
          echo number_format($stats['passes']) . ' (24h: ' . leadflow_common::trend_number_format($stats['trend_passes']) . ')';
        }
        break;
      case 'new_subscribers':
        if (empty($stats['new_subscribers'])) {
          echo '0';
        } else {
          echo $stats['new_subscribers'] . ' (24h: ' . leadflow_common::trend_number_format($stats['trend_new_subscribers']) . ')';
        }
        break;
    }
  } // manage_column_flow


  // additional option pages
  static function add_options_page() {
	
    add_menu_page('Dashboard', 'LeadsFlow Pro', 'manage_options', 'leadflow', array('leadflow_dashboard', 'content'), LF_PLUGIN_URL . '/images/menu-logo.png', 200);

    add_submenu_page('leadflow', 'Dashboard', 'Dashboard', 'manage_options', 'leadflow', array('leadflow_dashboard', 'content'));
    add_submenu_page('leadflow', 'Subscriber Tags', 'Subscriber Tags', 'manage_options', 'edit-tags.php?taxonomy=leadflow-subscriber-tags');
    add_submenu_page('leadflow', 'Statistics', 'Statistics', 'manage_options', 'leadflow-stats', array('leadflow_stats_view', 'content'));
    add_submenu_page('leadflow', 'Import', 'Import', 'manage_options', 'leadflow-import', array('leadflow_import', 'content'));
    add_submenu_page('leadflow', 'Settings', 'Settings', 'manage_options', 'leadflow-settings', array('leadflow_settings', 'content'));
	
	
    add_submenu_page(null, 'Bulk Process Information', 'Bulk Process Information', 'manage_options', 'leadflow-bulk', array('leadflow_bulk', 'content'));
    add_submenu_page(null, 'Export', 'Export', 'manage_options', 'leadflow-export', array('leadflow_export', 'content'));
  } // add_options_page


  // reorder submenu
  static function custom_menu_order($menu_order) {
    global $submenu, $menu;
    
	if(array_key_exists('leadflow',$submenu)){
		$tmp = $submenu['leadflow'];
	
		$submenu['leadflow'] = array();
		$submenu['leadflow'][] = $tmp[4];
		$submenu['leadflow'][] = $tmp[2];
		$submenu['leadflow'][] = $tmp[5];
		$submenu['leadflow'][] = $tmp[1];
		$submenu['leadflow'][] = $tmp[0];
		$submenu['leadflow'][] = $tmp[3];
		$submenu['leadflow'][] = $tmp[7];
		$submenu['leadflow'][] = $tmp[6];
		$submenu['leadflow'][] = $tmp[8];
		$submenu['leadflow'][] = $tmp[9];
	}
    return $menu_order;
  } // custom_menu_order


  // save optin meta
  // pregledat
  static function save_post_optin($post_id, $post, $update) {


    if (isset($_POST['optin-custom-url']) && $_POST[LF_META]['general']['custom_url'] == '1') {
      if (trim($_POST['optin-custom-url']) == '') {
        $url = '';
      } else {
        $url = trim(strtolower(sanitize_title($_POST['optin-custom-url'])), '/');
        if ($url) {
          $url = '/' . $url . '/';
        } else {
          $url = '/';
        }
      }

      update_post_meta($post_id, 'optin-custom-url', $url);
    } else {
      delete_post_meta($post_id, 'optin-custom-url');
    }

    // update system tags name
    $term = term_exists('optin-' . $post_id, 'leadflow-subscriber-system-tags', 'optins');
    if ($term) {
      wp_update_term((int) $term['term_id'], 'leadflow-subscriber-system-tags', array('name' => $post->post_title));
    }

    if ($_POST) {
      // On each save clear widget setup and re-save later
      update_post_meta($post_id, 'optin_widget_first_optin', '');
      update_post_meta($post_id, 'optin_widget_second_optin', '');
    }

    

    if ($parent_id = wp_is_post_revision($post_id)) {
      $post_id = $parent_id;
    }

    if (!empty($_POST[LF_META]) && $_POST[LF_META]) {
      $data = $_POST[LF_META];

      if (empty($data['general']['width'])) {
        $data['general']['width'] = '520px';
      }
	  
	  if (empty($data['general']['mode'])) {
        $data['general']['mode'] = 'simple';
      }

      update_post_meta($post_id, LF_META, $data);
    }

    if (!empty($_POST['_ab-test'])) {
      update_post_meta($post_id, '_ab-test', $_POST['_ab-test']);
    } else {
      delete_post_meta($post_id, '_ab-test');
    }

        
    if (!empty($_POST['optin_type'])) {
      update_post_meta($post_id, 'optin_type', $_POST['optin_type']);
    }
    if (!empty($_POST['optin_template'])) {
      update_post_meta($post_id, 'optin_template', $_POST['optin_template']);
    }

    if (!empty($_POST['optin_row'])) {
      $widgets = array();
      $row_settings = $_POST['optin_row'];
      $widget_settings = $_POST['optin_widget'];


      foreach ($row_settings as $box => $row) {
        $row_count = 0;

        foreach ($row as $row_id => $setup) {
          $settings = $setup['settings'];

          $widgets[$box][$row_id] = array();
          $widgets[$box][$row_id]['settings']['columns'] = $settings['columns'];
          $widgets[$box][$row_id]['settings']['full_width'] = $settings['full_width'];
          $widgets[$box][$row_id]['settings']['background_image'] = $settings['background_image'];
          $widgets[$box][$row_id]['settings']['background_color'] = $settings['background_color'];
          $widgets[$box][$row_id]['settings']['width'] = $settings['width'];

          if (!empty($widget_settings[$box])) {
            foreach ($widget_settings[$box] as $id => $settings) {
              if ($settings['container_id'] == $row_id) {
                // Clean text
                if (!empty($widget_settings[$box][$id]['settings']['text'])) {
                  $widget_settings[$box][$id]['settings']['text'] = trim($widget_settings[$box][$id]['settings']['text']);
                }
                
                // Paddings clean
                $widget_settings[$box][$id]['settings']['padding_left'] = self::check_paddings_margins($widget_settings[$box][$id]['settings']['padding_left']);
                $widget_settings[$box][$id]['settings']['padding_right'] = self::check_paddings_margins($widget_settings[$box][$id]['settings']['padding_right']);
                $widget_settings[$box][$id]['settings']['padding_top'] = self::check_paddings_margins($widget_settings[$box][$id]['settings']['padding_top']);
                $widget_settings[$box][$id]['settings']['padding_bottom'] = self::check_paddings_margins($widget_settings[$box][$id]['settings']['padding_bottom']);
                // Margins clean
                $widget_settings[$box][$id]['settings']['margin_left'] = self::check_paddings_margins($widget_settings[$box][$id]['settings']['margin_left']);
                $widget_settings[$box][$id]['settings']['margin_right'] = self::check_paddings_margins($widget_settings[$box][$id]['settings']['margin_right']);
                $widget_settings[$box][$id]['settings']['margin_top'] = self::check_paddings_margins($widget_settings[$box][$id]['settings']['margin_top']);
                $widget_settings[$box][$id]['settings']['margin_bottom'] = self::check_paddings_margins($widget_settings[$box][$id]['settings']['margin_bottom']);

                $widgets[$box][$row_id]['widgets'][] = $widget_settings[$box][$id];
                //var_dump($widget_settings[$box][$id]);
              }
            }
          }

        }
      }

      update_post_meta($post_id, 'optin_widget_first_optin', $widgets['first_optin']);
      update_post_meta($post_id, 'optin_widget_second_optin', $widgets['second_optin']);
    }

    if (!empty($_POST['lf-optin-preview'])) {
      set_transient('lf-show-preview', $post_id, 15);
    }
	
	// Load Predefined Optin Template Widgets and Settings
    if (!empty($_POST['optin_template'])) {
      $selected_template = $_POST['optin_template'];
      leadflow_optin_common::parse_template_layout($post_id, $selected_template);
    } 

  } // save_post_optin
  
  
  static function check_paddings_margins($value) {
    $value = rtrim($value, 'px');
    
    if (!is_numeric($value) && !ctype_digit($value)) {
      return '0';
    } else {
      return $value . 'px';
    }
  } // check_paddings_margins


  // save ab tests meta
  // pregledat
  static function save_ab_test($post_id, $post, $update) {
    if (isset($_POST['abtests-custom-url']) && $_POST[LF_META]['general']['custom_url'] == '1') {
      if (trim($_POST['abtests-custom-url']) == '') {
        $url = '';
      } else {
        $url = trim(strtolower(sanitize_title($_POST['abtests-custom-url'])), '/');
        if ($url) {
          $url = '/' . $url . '/';
        } else {
          $url = '/';
        }
      }
      update_post_meta($post_id, 'abtests-custom-url', $url);
    } else {
      delete_post_meta($post_id, 'abtests-custom-url');
    }

    if (!empty($_POST[LF_META])) {
      update_post_meta($post_id, LF_META, $_POST[LF_META]);
    }

    if (isset($_POST['ab_test_type'])) {
      update_post_meta($post_id, 'ab_test_type', $_POST['ab_test_type']);
    }
  } // save_ab_test


  // save subscriber meta
  static function save_post_subscribers($post_id, $post, $update) {
    if (!empty($_POST[LF_META])) {

      $meta = $_POST[LF_META];

      if (isset($meta['tags'])) {
        leadflow_subscribers_common::add_tags($post_id, $meta['tags']);
        unset($meta['tags']);
      }

      if (!empty($meta['editable'])) {
        foreach ($meta['editable'] as $key => $value) {
          update_post_meta($post_id, $key, $value);
        }
      }
    }
  } // save_post_subscribers


  // save flow meta
  static function save_post_flow($post_id, $post, $update) {
    // update system tags name
    $term = term_exists('flow-' . $post_id, 'leadflow-subscriber-system-tags', 'flows');
    if ($term) {
      wp_update_term((int) $term['term_id'], 'leadflow-subscriber-system-tags', array('name' => $post->post_title));
    }

    if (!empty($_POST[LF_META])) {
      update_post_meta($post_id, LF_META, $_POST[LF_META]);

      // if flow type changes components are reset
      if ($_POST['flow_type'] != leadflow_flow_common::get_flow_type($post_id)) {
        $components = array();
      } else {
        $components = $_POST['flow_components'];
      }

      if (empty($components) || !is_array($components)) {
        $components = array();
      }
      $new = array();
      //var_dump($components);
      foreach ($components as $comp_name => $comp_value) {
        // handle vars boxes
        if (!empty($comp_value['vars']) && is_array($comp_value['vars']['var_names'])) {
          $vars = array();
          foreach ($comp_value['vars']['var_names'] as $name) {
            if (!$name) {
              @next($comp_value['vars']['var_values']);
              continue;
            }
            $vars[$name] = current($comp_value['vars']['var_values']);
            next($comp_value['vars']['var_values']);
          } // foreach
          unset($comp_value['vars']['var_names'], $comp_value['vars']['var_values']);
          $comp_value['vars'] = $vars;
        }
        // handle possible non existing tags
        if (!empty($comp_value['tags_add'])) {
          $add_tags = array();
          foreach ($comp_value['tags_add'] as $tag_id) {
            if(!term_exists((int) $tag_id, 'leadflow-subscriber-tags')) {
              $new_tag = wp_insert_term($tag_id, 'leadflow-subscriber-tags', array('description' => 'Automatically created by Flow #' . $post_id));
              if (is_wp_error($new_tag) && $new_tag->get_error_data('term_exists')) {
                $add_tags[] = (int) $new_tag->get_error_data('term_exists');
              } elseif (!is_wp_error($new_tag)) {
                $add_tags[] = (int) $new_tag['term_id'];
              }
            } else {
              $add_tags[] = (int) $tag_id;
            }
          }
          $comp_value['tags_add'] = $add_tags;
        } // tags_add

        $new[] = $comp_value;
        //var_dump($comp_value); //die();
      } // foreach component

      update_post_meta($post_id, 'flow_components', $new);
    }

    if (isset($_POST['flow_type'])) {
      update_post_meta($post_id, 'flow_type', $_POST['flow_type']);
    }

    if (isset($_POST['flow-custom-url']) && $_POST[LF_META]['general']['custom_url'] == '1') {
      if (trim($_POST['flow-custom-url']) == '') {
        $url = '';
      } else {
        $url = trim(strtolower(sanitize_title($_POST['flow-custom-url'])), '/');
        if ($url) {
          $url = '/' . $url . '/';
        } else {
          $url = '/';
        }
      }
      update_post_meta($post_id, 'flow-custom-url', $url);
    } else {
      delete_post_meta($post_id, 'flow-custom-url');
    }
	
	
	if (isset($_POST['facebook-verify-token']) && $_POST[LF_META]['general']['facebook-leads'] == '1') {
      if (trim($_POST['facebook-verify-token']) == '') {
        $verify_token = '';
      } else {
        $verify_token = trim($_POST['facebook-verify-token']);        
      }
      update_post_meta($post_id, 'facebook-verify-token', $verify_token);
    } else {
      delete_post_meta($post_id, 'facebook-verify-token');
    }
	
	if (isset($_POST['facebook-connected-page-name']) && $_POST[LF_META]['general']['facebook-leads'] == '1') {
      if (trim($_POST['facebook-connected-page-name']) == '') {
        $page_name = '';
      } else {
        $page_name = trim($_POST['facebook-connected-page-name']);        
      }
      update_post_meta($post_id, 'facebook-connected-page-name', $page_name);
    } else {
      delete_post_meta($post_id, 'facebook-connected-page-name');
    }
	
	if (isset($_POST['facebook-connected-page-token']) && $_POST[LF_META]['general']['facebook-leads'] == '1') {
      if (trim($_POST['facebook-connected-page-token']) == '') {
        $page_token = '';
      } else {
        $page_token = trim($_POST['facebook-connected-page-token']);        
      }
      update_post_meta($post_id, 'facebook-connected-page-token', $page_token);
    } else {
      delete_post_meta($post_id, 'facebook-connected-page-token');
    }
	
  } // save_post_flow


  // save subscriber tags meta
  static function save_subscriber_tags_meta($term_id) {
    if ( isset( $_POST['term_meta'] ) ) {

      $term_meta = leadflow_common::get_subscriber_tags_meta($term_id);
      $cat_keys = array_keys($_POST['term_meta']);

      foreach ( $cat_keys as $key ) {
        if ( isset ( $_POST['term_meta'][$key] ) ) {
          $term_meta[$key] = $_POST['term_meta'][$key];
        }
      } // foreach

      // Save the option array.
      update_option('taxonomy_meta_' . $term_id, $term_meta);
    }
  } // save_subscriber_tags_meta


  // Add term page
  static function subscriber_tags_meta() {
    // this will add the custom meta field to the add new term page
    echo '<div class="form-field">
    <label for="term_meta[tag_color]">Tag Color:</label>
    <input type="text" name="term_meta[tag_color]" id="term_meta[tag_color]" value="" class="colorpicker">
    <p class="description">You can set custom background color for your tags.</p>
    </div>
    <div class="form-field">
    <label for="term_meta[font_color]">Font Color:</label>
    <input type="text" name="term_meta[font_color]" id="term_meta[font_color]" value="" class="colorpicker">
    <p class="description">You can set custom font color for your tags.</p>
    </div>';
  } // subscriber_tags_meta


  // Add term page
  static function edit_subscriber_tags_meta($term) {
    // this will add the custom meta field to the add new term page
    // retrieve the existing value(s) for this meta field. This returns an array
    $term_meta = leadflow_common::get_subscriber_tags_meta($term->term_id);

    echo '<tr class="form-field">
    <th scope="row" valign="top"><label for="term_meta[tag_color]">Tag Color:</label></th>
    <td>
    <input type="text" name="term_meta[tag_color]" id="term_meta[tag_color]" value="' . esc_attr( $term_meta['tag_color']) . '" class="colorpicker">
    <p class="description">You can set custom background color for your tags.</p>
    </td>
    </tr>
    <tr class="form-field">
    <th scope="row" valign="top"><label for="term_meta[font_color]">Font Color:</label></th>
    <td>
    <input type="text" name="term_meta[font_color]" id="term_meta[font_color]" value="' . esc_attr( $term_meta['font_color']) . '" class="colorpicker">
    <p class="description">You can set custom font color for your tags.</p>
    </td>
    </tr>';
  } // edit_subscriber_tags_meta


  // register meta boxes for pages and posts
  static function meta_boxes($post) {
    add_meta_box('lfp-lock', 'LeadsFlow Pro Content Locker', array('leadflow_page_lock', 'content'), 'page', 'normal', 'high');
    add_meta_box('lfp-lock', 'LeadsFlow Pro Content Locker', array('leadflow_page_lock', 'content'), 'post', 'normal', 'high');
  } // meta_boxes


  // register meta boxes for optins
  static function meta_boxes_optin($post) {
    $optin_type = leadflow_optin_common::get_optin_type($post->ID);

    self::remove_3rd_party_meta_boxes('optin-pages');
   
    if ($optin_type == 'embed') {
      add_meta_box('optin-embed', 'Embed Help', array('leadflow_optin_embed', 'content'), 'optin-pages', 'normal', 'high');
    }
	
	if ($optin_type == 'shortcode') {
      add_meta_box('optin-shortcode', 'Shortcode Help', array('leadflow_optin_shortcode', 'content'), 'optin-pages', 'normal', 'high');
    }
	
    add_meta_box('optin-type', 'Optin Type', array('leadflow_optin_type', 'content'), 'optin-pages', 'normal', 'high');
    add_meta_box('optin-template', 'Optin Template', array('leadflow_optin_template', 'content'), 'optin-pages', 'normal', 'high');

    if (!$optin_type) {
      return;
    }
	
    add_meta_box('optin-links', 'Navigation', array('leadflow_optin_links', 'content'), 'optin-pages', 'side');
    add_meta_box('optin-mode', 'Optin Mode', array('leadflow_optin_mode', 'content'), 'optin-pages', 'side');
    add_meta_box('optin-general', 'General Settings', array('leadflow_optin_general_settings', 'content'), 'optin-pages', 'normal', 'high');
    add_meta_box('optin-first-box', 'First Content Box', array('leadflow_optin_first_optin', 'content'), 'optin-pages', 'normal', 'high');
    add_meta_box('optin-second-box', 'Second Content Box', array('leadflow_optin_second_optin', 'content'), 'optin-pages', 'normal', 'high');
    add_meta_box('optin-form', 'Form Settings', array('leadflow_optin_form', 'content'), 'optin-pages', 'normal', 'high');

    if ($optin_type == 'standalone' ) {
      add_meta_box('optin-background', 'Background', array('leadflow_optin_background', 'content'), 'optin-pages', 'normal', 'high');
    } elseif ($optin_type == 'popup') {
      add_meta_box('optin-popup', 'Popup triggers', array('leadflow_optin_popup', 'content'), 'optin-pages', 'normal', 'high');
    } elseif ($optin_type == 'embed') {
	  add_meta_box('optin-background', 'Background', array('leadflow_optin_background', 'content'), 'optin-pages', 'normal', 'high');
    } elseif ($optin_type == 'pullout') {
      add_meta_box('optin-pullout', 'Pullout triggers', array('leadflow_optin_pullout', 'content'), 'optin-pages', 'normal', 'high');
    } elseif ($optin_type == 'header') {
      add_meta_box('optin-background', 'Background', array('leadflow_optin_background', 'content'), 'optin-pages', 'normal', 'high');
      add_meta_box('optin-header', 'Header/Footer Setup', array('leadflow_optin_header', 'content'), 'optin-pages', 'normal', 'high');
    } elseif ($optin_type == 'welcome') {
      add_meta_box('optin-welcome', 'Welcome Splash triggers', array('leadflow_optin_welcome', 'content'), 'optin-pages', 'normal', 'high');
      add_meta_box('optin-background', 'Background', array('leadflow_optin_background', 'content'), 'optin-pages', 'normal', 'high');
    } else {
      // unknown optin type
    }

  } // meta_boxes_optin


  // register meta boxes for flows
  static function meta_boxes_flow($post) {
    $flow_type = leadflow_flow_common::get_flow_type($post->ID);

    self::remove_3rd_party_meta_boxes('lead-flows');

    add_meta_box('flow-type', 'Flow Type', array('leadflow_flow_type', 'content'), 'lead-flows', 'normal', 'high');

    if (!$flow_type) {
      return;
    }
    add_meta_box('flow-links', 'Navigation', array('leadflow_flow_links', 'content'), 'lead-flows', 'side');
    if ($flow_type != 'internal') {
      add_meta_box('flow-general', 'General Settings', array('leadflow_flow_general_settings', 'content'), 'lead-flows', 'normal', 'high');
    }
    if ($flow_type == 'form') {
      add_meta_box('flow-form', 'Form Catcher Settings', array('leadflow_flow_form', 'content'), 'lead-flows', 'normal', 'high');
    }
    add_meta_box('flow-components', 'Flow Components', array('leadflow_flow_components', 'content'), 'lead-flows', 'normal', 'high');
    add_meta_box('flow-debug', 'Debugging', array('leadflow_flow_debug', 'content'), 'lead-flows', 'normal', 'high');
  } // meta_boxes_flow


  // register meta boxes for subscribers
  static function meta_boxes_subscibers($post) {
    // remove wp default tag box
    remove_meta_box('tagsdiv-leadflow-subscriber-tags', 'leadflow-subscribers', 'side');
    // add our own tag box
    add_meta_box('subscriber_tags', 'Subscriber Tags', array('leadflow_subscribers_metabox', 'tags_box'), 'leadflow-subscribers', 'side', 'high');
    // show user data
    add_meta_box('subscriber_graphic_data', 'Information', array('leadflow_subscribers_metabox', 'graphic_info'), 'leadflow-subscribers', 'normal', 'high');
    add_meta_box('subscriber_editable_data', 'Edit Basic Info', array('leadflow_subscribers_metabox', 'editable'), 'leadflow-subscribers', 'normal', 'high');

    add_meta_box('subscriber_custom_vars', 'Custom Variables', array('leadflow_subscribers_metabox', 'custom_vars'), 'leadflow-subscribers', 'normal', 'high');

    add_meta_box('subscriber_events', 'Events', array('leadflow_subscribers_metabox', 'events'), 'leadflow-subscribers', 'normal', 'high');

    add_meta_box('subscriber_fullcontact', 'FullContact Detailed Info', array('leadflow_subscribers_metabox', 'fullcontact'), 'leadflow-subscribers', 'normal', 'high');
  } // meta_boxes_subscibers


  // register meta boxes for ab tests
  static function meta_boxes_abtests($post) {

    add_meta_box('abtest_type', 'A/B Test Type', array('leadflow_abtests_metabox', 'type_box'), 'leadflow-abtests', 'normal', 'high');

    $ab_type = leadflow_ab_test_common::get_ab_test_type($post->ID);

    if ($ab_type) {
      add_meta_box('abtest-links', 'Navigation', array('leadflow_ab_links', 'content'), 'leadflow-abtests', 'side');
      if ($ab_type == 'standalone') {
        add_meta_box('abtest-general', 'A/B Test General Settings', array('leadflow_ab_general_settings', 'content'), 'leadflow-abtests', 'normal', 'high');
      }
	  if ($ab_type == 'shortcode') {
        add_meta_box('abtest-shortcode', 'Shortcode Help', array('leadflow_ab_shortcode', 'content'), 'leadflow-abtests', 'normal', 'high');
      }
	  
	  if ($ab_type == 'popup' || $ab_type == 'header' || $ab_type == 'welcome'  || $ab_type == 'pullout' ) {
      	add_meta_box('abtest-conditionals', 'A/B Test Conditionals', array('leadflow_abtests_metabox', 'conditionals_box'), 'leadflow-abtests', 'normal', 'high');
	  }
    }
  } // meta_boxes_subscibers


  // removes all 3rd party meta boxes
  static function remove_3rd_party_meta_boxes($screen) {
    global $wp_meta_boxes;
    $remove = array();

    if (isset($wp_meta_boxes['lead-flows']['normal']['high']) && is_array($wp_meta_boxes['lead-flows']['normal']['high'])) {
      foreach ($wp_meta_boxes['lead-flows']['normal']['high'] as $box) {
        if (strpos($box['id'], 'optin') === false && strpos($box['id'], 'flow') === false && strpos($box['id'], 'wpseo_meta') === false) {
          $remove[] = array('id' => $box['id'], 'screen' => $screen, 'context' => 'normal');
        }
      } // foreach normal-high
    }
    if (isset($wp_meta_boxes['lead-flows']['normal']['low']) && is_array($wp_meta_boxes['lead-flows']['normal']['low'])) {
      foreach ($wp_meta_boxes['lead-flows']['normal']['low'] as $box) {
        if (strpos($box['id'], 'optin') === false && strpos($box['id'], 'flow') === false && strpos($box['id'], 'wpseo_meta') === false) {
          $remove[] = array('id' => $box['id'], 'screen' => $screen, 'context' => 'normal');
        }
      } // foreach normal-low
    }
    if (isset($wp_meta_boxes['lead-flows']['normal']['default']) && is_array($wp_meta_boxes['lead-flows']['normal']['default'])) {
      foreach ($wp_meta_boxes['lead-flows']['normal']['default'] as $box) {
        if (strpos($box['id'], 'optin') === false && strpos($box['id'], 'flow') === false && strpos($box['id'], 'wpseo_meta') === false) {
          $remove[] = array('id' => $box['id'], 'screen' => $screen, 'context' => 'normal');
        }
      } // foreach normal-default
    }

    foreach ($remove as $tmp) {
      remove_meta_box($tmp['id'], $tmp['screen'], $tmp['context']);
    }
  } // remove_3rd_party_meta_boxes

	
	
  // save list of google font in options
  static function load_google_fonts() {
	 $google_fonts_list=unserialize('a:733:{s:9:"open_sans";a:3:{s:4:"name";s:9:"Open Sans";s:8:"variants";s:70:"300,300italic,regular,italic,600,600italic,700,700italic,800,800italic";s:8:"fallback";s:10:"sans-serif";}s:6:"roboto";a:3:{s:4:"name";s:6:"Roboto";s:8:"variants";s:84:"100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:4:"lato";a:3:{s:4:"name";s:4:"Lato";s:8:"variants";s:70:"100,100italic,300,300italic,regular,italic,700,700italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:10:"slabo_27px";a:3:{s:4:"name";s:10:"Slabo 27px";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:16:"roboto_condensed";a:3:{s:4:"name";s:16:"Roboto Condensed";s:8:"variants";s:42:"300,300italic,regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:6:"oswald";a:3:{s:4:"name";s:6:"Oswald";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:10:"sans-serif";}s:10:"montserrat";a:3:{s:4:"name";s:10:"Montserrat";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:15:"source_sans_pro";a:3:{s:4:"name";s:15:"Source Sans Pro";s:8:"variants";s:84:"200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:4:"lora";a:3:{s:4:"name";s:4:"Lora";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:7:"raleway";a:3:{s:4:"name";s:7:"Raleway";s:8:"variants";s:126:"100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:7:"pt_sans";a:3:{s:4:"name";s:7:"PT Sans";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:19:"open_sans_condensed";a:3:{s:4:"name";s:19:"Open Sans Condensed";s:8:"variants";s:17:"300,300italic,700";s:8:"fallback";s:10:"sans-serif";}s:10:"droid_sans";a:3:{s:4:"name";s:10:"Droid Sans";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:6:"ubuntu";a:3:{s:4:"name";s:6:"Ubuntu";s:8:"variants";s:56:"300,300italic,regular,italic,500,500italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:11:"roboto_slab";a:3:{s:4:"name";s:11:"Roboto Slab";s:8:"variants";s:19:"100,300,regular,700";s:8:"fallback";s:5:"serif";}s:11:"droid_serif";a:3:{s:4:"name";s:11:"Droid Serif";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:5:"arimo";a:3:{s:4:"name";s:5:"Arimo";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:10:"fjalla_one";a:3:{s:4:"name";s:10:"Fjalla One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:14:"pt_sans_narrow";a:3:{s:4:"name";s:14:"PT Sans Narrow";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:12:"merriweather";a:3:{s:4:"name";s:12:"Merriweather";s:8:"variants";s:56:"300,300italic,regular,italic,700,700italic,900,900italic";s:8:"fallback";s:5:"serif";}s:9:"noto_sans";a:3:{s:4:"name";s:9:"Noto Sans";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:13:"titillium_web";a:3:{s:4:"name";s:13:"Titillium Web";s:8:"variants";s:74:"200,200italic,300,300italic,regular,italic,600,600italic,700,700italic,900";s:8:"fallback";s:10:"sans-serif";}s:13:"alegreya_sans";a:3:{s:4:"name";s:13:"Alegreya Sans";s:8:"variants";s:98:"100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,800,800italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:8:"pt_serif";a:3:{s:4:"name";s:8:"PT Serif";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:11:"passion_one";a:3:{s:4:"name";s:11:"Passion One";s:8:"variants";s:15:"regular,700,900";s:8:"fallback";s:7:"display";}s:10:"poiret_one";a:3:{s:4:"name";s:10:"Poiret One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"candal";a:3:{s:4:"name";s:6:"Candal";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:16:"playfair_display";a:3:{s:4:"name";s:16:"Playfair Display";s:8:"variants";s:42:"regular,italic,700,700italic,900,900italic";s:8:"fallback";s:5:"serif";}s:12:"indie_flower";a:3:{s:4:"name";s:12:"Indie Flower";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"bitter";a:3:{s:4:"name";s:6:"Bitter";s:8:"variants";s:18:"regular,italic,700";s:8:"fallback";s:5:"serif";}s:5:"dosis";a:3:{s:4:"name";s:5:"Dosis";s:8:"variants";s:31:"200,300,regular,500,600,700,800";s:8:"fallback";s:10:"sans-serif";}s:5:"cabin";a:3:{s:4:"name";s:5:"Cabin";s:8:"variants";s:56:"regular,italic,500,500italic,600,600italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:17:"yanone_kaffeesatz";a:3:{s:4:"name";s:17:"Yanone Kaffeesatz";s:8:"variants";s:19:"200,300,regular,700";s:8:"fallback";s:10:"sans-serif";}s:6:"oxygen";a:3:{s:4:"name";s:6:"Oxygen";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:10:"sans-serif";}s:7:"lobster";a:3:{s:4:"name";s:7:"Lobster";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:4:"hind";a:3:{s:4:"name";s:4:"Hind";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:5:"anton";a:3:{s:4:"name";s:5:"Anton";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:4:"arvo";a:3:{s:4:"name";s:4:"Arvo";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:10:"noto_serif";a:3:{s:4:"name";s:10:"Noto Serif";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:6:"nunito";a:3:{s:4:"name";s:6:"Nunito";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:10:"sans-serif";}s:11:"inconsolata";a:3:{s:4:"name";s:11:"Inconsolata";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:9:"monospace";}s:4:"abel";a:3:{s:4:"name";s:4:"Abel";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:10:"bree_serif";a:3:{s:4:"name";s:10:"Bree Serif";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:4:"muli";a:3:{s:4:"name";s:4:"Muli";s:8:"variants";s:28:"300,300italic,regular,italic";s:8:"fallback";s:10:"sans-serif";}s:9:"fira_sans";a:3:{s:4:"name";s:9:"Fira Sans";s:8:"variants";s:56:"300,300italic,regular,italic,500,500italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:12:"josefin_sans";a:3:{s:4:"name";s:12:"Josefin Sans";s:8:"variants";s:70:"100,100italic,300,300italic,regular,italic,600,600italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:16:"ubuntu_condensed";a:3:{s:4:"name";s:16:"Ubuntu Condensed";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"cinzel";a:3:{s:4:"name";s:6:"Cinzel";s:8:"variants";s:15:"regular,700,900";s:8:"fallback";s:5:"serif";}s:17:"libre_baskerville";a:3:{s:4:"name";s:17:"Libre Baskerville";s:8:"variants";s:18:"regular,italic,700";s:8:"fallback";s:5:"serif";}s:5:"exo_2";a:3:{s:4:"name";s:5:"Exo 2";s:8:"variants";s:126:"100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:8:"pacifico";a:3:{s:4:"name";s:8:"Pacifico";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:18:"shadows_into_light";a:3:{s:4:"name";s:18:"Shadows Into Light";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:4:"play";a:3:{s:4:"name";s:4:"Play";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:12:"crimson_text";a:3:{s:4:"name";s:12:"Crimson Text";s:8:"variants";s:42:"regular,italic,600,600italic,700,700italic";s:8:"fallback";s:5:"serif";}s:4:"asap";a:3:{s:4:"name";s:4:"Asap";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:7:"signika";a:3:{s:4:"name";s:7:"Signika";s:8:"variants";s:19:"300,regular,600,700";s:8:"fallback";s:10:"sans-serif";}s:6:"cuprum";a:3:{s:4:"name";s:6:"Cuprum";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:14:"archivo_narrow";a:3:{s:4:"name";s:14:"Archivo Narrow";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:12:"francois_one";a:3:{s:4:"name";s:12:"Francois One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"quicksand";a:3:{s:4:"name";s:9:"Quicksand";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:10:"sans-serif";}s:17:"merriweather_sans";a:3:{s:4:"name";s:17:"Merriweather Sans";s:8:"variants";s:56:"300,300italic,regular,italic,700,700italic,800,800italic";s:8:"fallback";s:10:"sans-serif";}s:8:"alegreya";a:3:{s:4:"name";s:8:"Alegreya";s:8:"variants";s:42:"regular,italic,700,700italic,900,900italic";s:8:"fallback";s:5:"serif";}s:9:"amatic_sc";a:3:{s:4:"name";s:9:"Amatic SC";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:11:"handwriting";}s:8:"vollkorn";a:3:{s:4:"name";s:8:"Vollkorn";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:9:"maven_pro";a:3:{s:4:"name";s:9:"Maven Pro";s:8:"variants";s:19:"regular,500,700,900";s:8:"fallback";s:10:"sans-serif";}s:8:"orbitron";a:3:{s:4:"name";s:8:"Orbitron";s:8:"variants";s:19:"regular,500,700,900";s:8:"fallback";s:10:"sans-serif";}s:12:"varela_round";a:3:{s:4:"name";s:12:"Varela Round";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"karla";a:3:{s:4:"name";s:5:"Karla";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:9:"righteous";a:3:{s:4:"name";s:9:"Righteous";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"dancing_script";a:3:{s:4:"name";s:14:"Dancing Script";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:11:"handwriting";}s:3:"exo";a:3:{s:4:"name";s:3:"Exo";s:8:"variants";s:126:"100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:7:"rokkitt";a:3:{s:4:"name";s:7:"Rokkitt";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:19:"architects_daughter";a:3:{s:4:"name";s:19:"Architects Daughter";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:9:"questrial";a:3:{s:4:"name";s:9:"Questrial";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"chewy";a:3:{s:4:"name";s:5:"Chewy";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:15:"pt_sans_caption";a:3:{s:4:"name";s:15:"PT Sans Caption";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:7:"bangers";a:3:{s:4:"name";s:7:"Bangers";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"abril_fatface";a:3:{s:4:"name";s:13:"Abril Fatface";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"patua_one";a:3:{s:4:"name";s:9:"Patua One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"sigmar_one";a:3:{s:4:"name";s:10:"Sigmar One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:18:"pathway_gothic_one";a:3:{s:4:"name";s:18:"Pathway Gothic One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"monda";a:3:{s:4:"name";s:5:"Monda";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:17:"quattrocento_sans";a:3:{s:4:"name";s:17:"Quattrocento Sans";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:12:"josefin_slab";a:3:{s:4:"name";s:12:"Josefin Slab";s:8:"variants";s:70:"100,100italic,300,300italic,regular,italic,600,600italic,700,700italic";s:8:"fallback";s:5:"serif";}s:9:"russo_one";a:3:{s:4:"name";s:9:"Russo One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"benchnine";a:3:{s:4:"name";s:9:"BenchNine";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:10:"sans-serif";}s:9:"ropa_sans";a:3:{s:4:"name";s:9:"Ropa Sans";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:10:"sans-serif";}s:9:"comfortaa";a:3:{s:4:"name";s:9:"Comfortaa";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:7:"display";}s:10:"news_cycle";a:3:{s:4:"name";s:10:"News Cycle";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:11:"lobster_two";a:3:{s:4:"name";s:11:"Lobster Two";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:7:"display";}s:11:"crete_round";a:3:{s:4:"name";s:11:"Crete Round";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:14:"kaushan_script";a:3:{s:4:"name";s:14:"Kaushan Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"eb_garamond";a:3:{s:4:"name";s:11:"EB Garamond";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"istok_web";a:3:{s:4:"name";s:9:"Istok Web";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:5:"gudea";a:3:{s:4:"name";s:5:"Gudea";s:8:"variants";s:18:"regular,italic,700";s:8:"fallback";s:10:"sans-serif";}s:7:"abeezee";a:3:{s:4:"name";s:7:"ABeeZee";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:10:"sans-serif";}s:12:"pontano_sans";a:3:{s:4:"name";s:12:"Pontano Sans";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:21:"covered_by_your_grace";a:3:{s:4:"name";s:21:"Covered By Your Grace";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"noticia_text";a:3:{s:4:"name";s:12:"Noticia Text";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:17:"gloria_hallelujah";a:3:{s:4:"name";s:17:"Gloria Hallelujah";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:13:"didact_gothic";a:3:{s:4:"name";s:13:"Didact Gothic";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:11:"fredoka_one";a:3:{s:4:"name";s:11:"Fredoka One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"cantarell";a:3:{s:4:"name";s:9:"Cantarell";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:15:"hammersmith_one";a:3:{s:4:"name";s:15:"Hammersmith One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:11:"philosopher";a:3:{s:4:"name";s:11:"Philosopher";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:11:"coming_soon";a:3:{s:4:"name";s:11:"Coming Soon";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:15:"old_standard_tt";a:3:{s:4:"name";s:15:"Old Standard TT";s:8:"variants";s:18:"regular,italic,700";s:8:"fallback";s:5:"serif";}s:6:"armata";a:3:{s:4:"name";s:6:"Armata";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"rambla";a:3:{s:4:"name";s:6:"Rambla";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:13:"archivo_black";a:3:{s:4:"name";s:13:"Archivo Black";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"domine";a:3:{s:4:"name";s:6:"Domine";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:9:"tangerine";a:3:{s:4:"name";s:9:"Tangerine";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:11:"handwriting";}s:9:"courgette";a:3:{s:4:"name";s:9:"Courgette";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"satisfy";a:3:{s:4:"name";s:7:"Satisfy";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"tinos";a:3:{s:4:"name";s:5:"Tinos";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:15:"cabin_condensed";a:3:{s:4:"name";s:15:"Cabin Condensed";s:8:"variants";s:19:"regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:5:"kreon";a:3:{s:4:"name";s:5:"Kreon";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:5:"serif";}s:7:"sanchez";a:3:{s:4:"name";s:7:"Sanchez";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:4:"ruda";a:3:{s:4:"name";s:4:"Ruda";s:8:"variants";s:15:"regular,700,900";s:8:"fallback";s:10:"sans-serif";}s:7:"handlee";a:3:{s:4:"name";s:7:"Handlee";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"cookie";a:3:{s:4:"name";s:6:"Cookie";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:15:"source_code_pro";a:3:{s:4:"name";s:15:"Source Code Pro";s:8:"variants";s:31:"200,300,regular,500,600,700,900";s:8:"fallback";s:9:"monospace";}s:6:"varela";a:3:{s:4:"name";s:6:"Varela";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"economica";a:3:{s:4:"name";s:9:"Economica";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:4:"acme";a:3:{s:4:"name";s:4:"Acme";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"rock_salt";a:3:{s:4:"name";s:9:"Rock Salt";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:13:"alfa_slab_one";a:3:{s:4:"name";s:13:"Alfa Slab One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"quattrocento";a:3:{s:4:"name";s:12:"Quattrocento";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:7:"poppins";a:3:{s:4:"name";s:7:"Poppins";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:16:"permanent_marker";a:3:{s:4:"name";s:16:"Permanent Marker";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:4:"jura";a:3:{s:4:"name";s:4:"Jura";s:8:"variants";s:19:"300,regular,500,600";s:8:"fallback";s:10:"sans-serif";}s:18:"gentium_book_basic";a:3:{s:4:"name";s:18:"Gentium Book Basic";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:8:"voltaire";a:3:{s:4:"name";s:8:"Voltaire";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:12:"luckiest_guy";a:3:{s:4:"name";s:12:"Luckiest Guy";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"rubik";a:3:{s:4:"name";s:5:"Rubik";s:8:"variants";s:70:"300,300italic,regular,italic,500,500italic,700,700italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:9:"work_sans";a:3:{s:4:"name";s:9:"Work Sans";s:8:"variants";s:39:"100,200,300,regular,500,600,700,800,900";s:8:"fallback";s:10:"sans-serif";}s:7:"sintony";a:3:{s:4:"name";s:7:"Sintony";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:22:"shadows_into_light_two";a:3:{s:4:"name";s:22:"Shadows Into Light Two";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"roboto_mono";a:3:{s:4:"name";s:11:"Roboto Mono";s:8:"variants";s:70:"100,100italic,300,300italic,regular,italic,500,500italic,700,700italic";s:8:"fallback";s:9:"monospace";}s:5:"bevan";a:3:{s:4:"name";s:5:"Bevan";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"fauna_one";a:3:{s:4:"name";s:9:"Fauna One";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"cardo";a:3:{s:4:"name";s:5:"Cardo";s:8:"variants";s:18:"regular,italic,700";s:8:"fallback";s:5:"serif";}s:11:"paytone_one";a:3:{s:4:"name";s:11:"Paytone One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:13:"pinyon_script";a:3:{s:4:"name";s:13:"Pinyon Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"actor";a:3:{s:4:"name";s:5:"Actor";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:13:"special_elite";a:3:{s:4:"name";s:13:"Special Elite";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"playball";a:3:{s:4:"name";s:8:"Playball";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"volkhov";a:3:{s:4:"name";s:7:"Volkhov";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:8:"vidaloka";a:3:{s:4:"name";s:8:"Vidaloka";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:10:"bad_script";a:3:{s:4:"name";s:10:"Bad Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"great_vibes";a:3:{s:4:"name";s:11:"Great Vibes";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:10:"changa_one";a:3:{s:4:"name";s:10:"Changa One";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:7:"display";}s:8:"amaranth";a:3:{s:4:"name";s:8:"Amaranth";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:11:"oleo_script";a:3:{s:4:"name";s:11:"Oleo Script";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:9:"audiowide";a:3:{s:4:"name";s:9:"Audiowide";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"antic_slab";a:3:{s:4:"name";s:10:"Antic Slab";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:12:"marck_script";a:3:{s:4:"name";s:12:"Marck Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"basic";a:3:{s:4:"name";s:5:"Basic";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"enriqueta";a:3:{s:4:"name";s:9:"Enriqueta";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:19:"playfair_display_sc";a:3:{s:4:"name";s:19:"Playfair Display SC";s:8:"variants";s:42:"regular,italic,700,700italic,900,900italic";s:8:"fallback";s:5:"serif";}s:6:"arapey";a:3:{s:4:"name";s:6:"Arapey";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:8:"boogaloo";a:3:{s:4:"name";s:8:"Boogaloo";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"amiri";a:3:{s:4:"name";s:5:"Amiri";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:6:"nobile";a:3:{s:4:"name";s:6:"Nobile";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:7:"niconne";a:3:{s:4:"name";s:7:"Niconne";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:15:"droid_sans_mono";a:3:{s:4:"name";s:15:"Droid Sans Mono";s:8:"variants";s:7:"regular";s:8:"fallback";s:9:"monospace";}s:16:"sorts_mill_goudy";a:3:{s:4:"name";s:16:"Sorts Mill Goudy";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:9:"fugaz_one";a:3:{s:4:"name";s:9:"Fugaz One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"chivo";a:3:{s:4:"name";s:5:"Chivo";s:8:"variants";s:28:"regular,italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:10:"squada_one";a:3:{s:4:"name";s:10:"Squada One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"molengo";a:3:{s:4:"name";s:7:"Molengo";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"khand";a:3:{s:4:"name";s:5:"Khand";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:14:"calligraffitti";a:3:{s:4:"name";s:14:"Calligraffitti";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"damion";a:3:{s:4:"name";s:6:"Damion";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:10:"doppio_one";a:3:{s:4:"name";s:10:"Doppio One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:4:"viga";a:3:{s:4:"name";s:4:"Viga";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"limelight";a:3:{s:4:"name";s:9:"Limelight";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"scada";a:3:{s:4:"name";s:5:"Scada";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:5:"copse";a:3:{s:4:"name";s:5:"Copse";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:16:"signika_negative";a:3:{s:4:"name";s:16:"Signika Negative";s:8:"variants";s:19:"300,regular,600,700";s:8:"fallback";s:10:"sans-serif";}s:5:"share";a:3:{s:4:"name";s:5:"Share";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:7:"display";}s:8:"marmelad";a:3:{s:4:"name";s:8:"Marmelad";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:17:"just_another_hand";a:3:{s:4:"name";s:17:"Just Another Hand";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:10:"gochi_hand";a:3:{s:4:"name";s:10:"Gochi Hand";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"glegoo";a:3:{s:4:"name";s:6:"Glegoo";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:8:"overlock";a:3:{s:4:"name";s:8:"Overlock";s:8:"variants";s:42:"regular,italic,700,700italic,900,900italic";s:8:"fallback";s:7:"display";}s:8:"days_one";a:3:{s:4:"name";s:8:"Days One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"lusitana";a:3:{s:4:"name";s:8:"Lusitana";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:4:"coda";a:3:{s:4:"name";s:4:"Coda";s:8:"variants";s:11:"regular,800";s:8:"fallback";s:7:"display";}s:10:"jockey_one";a:3:{s:4:"name";s:10:"Jockey One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"carme";a:3:{s:4:"name";s:5:"Carme";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:16:"alegreya_sans_sc";a:3:{s:4:"name";s:16:"Alegreya Sans SC";s:8:"variants";s:98:"100,100italic,300,300italic,regular,italic,500,500italic,700,700italic,800,800italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:5:"ultra";a:3:{s:4:"name";s:5:"Ultra";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:12:"crafty_girls";a:3:{s:4:"name";s:12:"Crafty Girls";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:14:"homemade_apple";a:3:{s:4:"name";s:14:"Homemade Apple";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"electrolize";a:3:{s:4:"name";s:11:"Electrolize";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"alice";a:3:{s:4:"name";s:5:"Alice";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:16:"source_serif_pro";a:3:{s:4:"name";s:16:"Source Serif Pro";s:8:"variants";s:15:"regular,600,700";s:8:"fallback";s:5:"serif";}s:21:"montserrat_alternates";a:3:{s:4:"name";s:21:"Montserrat Alternates";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:6:"neuton";a:3:{s:4:"name";s:6:"Neuton";s:8:"variants";s:30:"200,300,regular,italic,700,800";s:8:"fallback";s:5:"serif";}s:13:"black_ops_one";a:3:{s:4:"name";s:13:"Black Ops One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"cantata_one";a:3:{s:4:"name";s:11:"Cantata One";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:12:"contrail_one";a:3:{s:4:"name";s:12:"Contrail One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"patrick_hand";a:3:{s:4:"name";s:12:"Patrick Hand";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:9:"spinnaker";a:3:{s:4:"name";s:9:"Spinnaker";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"michroma";a:3:{s:4:"name";s:8:"Michroma";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"nixie_one";a:3:{s:4:"name";s:9:"Nixie One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:15:"walter_turncoat";a:3:{s:4:"name";s:15:"Walter Turncoat";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:8:"quantico";a:3:{s:4:"name";s:8:"Quantico";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:23:"waiting_for_the_sunrise";a:3:{s:4:"name";s:23:"Waiting for the Sunrise";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"kanit";a:3:{s:4:"name";s:5:"Kanit";s:8:"variants";s:126:"100,100italic,200,200italic,300,300italic,regular,italic,500,500italic,600,600italic,700,700italic,800,800italic,900,900italic";s:8:"fallback";s:10:"sans-serif";}s:5:"antic";a:3:{s:4:"name";s:5:"Antic";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:17:"cherry_cream_soda";a:3:{s:4:"name";s:17:"Cherry Cream Soda";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"allerta";a:3:{s:4:"name";s:7:"Allerta";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"rajdhani";a:3:{s:4:"name";s:8:"Rajdhani";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:7:"average";a:3:{s:4:"name";s:7:"Average";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:10:"alex_brush";a:3:{s:4:"name";s:10:"Alex Brush";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:9:"syncopate";a:3:{s:4:"name";s:9:"Syncopate";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:7:"aldrich";a:3:{s:4:"name";s:7:"Aldrich";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:14:"bubblegum_sans";a:3:{s:4:"name";s:14:"Bubblegum Sans";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:15:"julius_sans_one";a:3:{s:4:"name";s:15:"Julius Sans One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:10:"sacramento";a:3:{s:4:"name";s:10:"Sacramento";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:20:"nothing_you_could_do";a:3:{s:4:"name";s:20:"Nothing You Could Do";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:9:"catamaran";a:3:{s:4:"name";s:9:"Catamaran";s:8:"variants";s:39:"100,200,300,regular,500,600,700,800,900";s:8:"fallback";s:10:"sans-serif";}s:6:"marvel";a:3:{s:4:"name";s:6:"Marvel";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:8:"homenaje";a:3:{s:4:"name";s:8:"Homenaje";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"khula";a:3:{s:4:"name";s:5:"Khula";s:8:"variants";s:23:"300,regular,600,700,800";s:8:"fallback";s:10:"sans-serif";}s:15:"allerta_stencil";a:3:{s:4:"name";s:15:"Allerta Stencil";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"magra";a:3:{s:4:"name";s:5:"Magra";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:5:"kalam";a:3:{s:4:"name";s:5:"Kalam";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:11:"handwriting";}s:11:"ceviche_one";a:3:{s:4:"name";s:11:"Ceviche One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"marcellus";a:3:{s:4:"name";s:9:"Marcellus";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:13:"gentium_basic";a:3:{s:4:"name";s:13:"Gentium Basic";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:4:"teko";a:3:{s:4:"name";s:4:"Teko";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:20:"fredericka_the_great";a:3:{s:4:"name";s:20:"Fredericka the Great";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"ubuntu_mono";a:3:{s:4:"name";s:11:"Ubuntu Mono";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:9:"monospace";}s:7:"kameron";a:3:{s:4:"name";s:7:"Kameron";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:12:"finger_paint";a:3:{s:4:"name";s:12:"Finger Paint";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:16:"fontdiner_swanky";a:3:{s:4:"name";s:16:"Fontdiner Swanky";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"six_caps";a:3:{s:4:"name";s:8:"Six Caps";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"pt_mono";a:3:{s:4:"name";s:7:"PT Mono";s:8:"variants";s:7:"regular";s:8:"fallback";s:9:"monospace";}s:13:"reenie_beanie";a:3:{s:4:"name";s:13:"Reenie Beanie";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:16:"pt_serif_caption";a:3:{s:4:"name";s:16:"PT Serif Caption";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:15:"berkshire_swash";a:3:{s:4:"name";s:15:"Berkshire Swash";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"allura";a:3:{s:4:"name";s:6:"Allura";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"puritan";a:3:{s:4:"name";s:7:"Puritan";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:6:"halant";a:3:{s:4:"name";s:6:"Halant";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:5:"serif";}s:6:"rancho";a:3:{s:4:"name";s:6:"Rancho";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"radley";a:3:{s:4:"name";s:6:"Radley";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:5:"prata";a:3:{s:4:"name";s:5:"Prata";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"telex";a:3:{s:4:"name";s:5:"Telex";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:10:"yellowtail";a:3:{s:4:"name";s:10:"Yellowtail";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"freckle_face";a:3:{s:4:"name";s:12:"Freckle Face";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"chelsea_market";a:3:{s:4:"name";s:14:"Chelsea Market";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"coustard";a:3:{s:4:"name";s:8:"Coustard";s:8:"variants";s:11:"regular,900";s:8:"fallback";s:5:"serif";}s:14:"carrois_gothic";a:3:{s:4:"name";s:14:"Carrois Gothic";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:10:"advent_pro";a:3:{s:4:"name";s:10:"Advent Pro";s:8:"variants";s:31:"100,200,300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:8:"merienda";a:3:{s:4:"name";s:8:"Merienda";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:11:"handwriting";}s:6:"neucha";a:3:{s:4:"name";s:6:"Neucha";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:13:"mouse_memoirs";a:3:{s:4:"name";s:13:"Mouse Memoirs";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"montez";a:3:{s:4:"name";s:6:"Montez";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"average_sans";a:3:{s:4:"name";s:12:"Average Sans";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"adamina";a:3:{s:4:"name";s:7:"Adamina";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"rochester";a:3:{s:4:"name";s:9:"Rochester";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"lekton";a:3:{s:4:"name";s:6:"Lekton";s:8:"variants";s:18:"regular,italic,700";s:8:"fallback";s:10:"sans-serif";}s:6:"cambay";a:3:{s:4:"name";s:6:"Cambay";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:12:"cabin_sketch";a:3:{s:4:"name";s:12:"Cabin Sketch";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:6:"cutive";a:3:{s:4:"name";s:6:"Cutive";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"kadwa";a:3:{s:4:"name";s:5:"Kadwa";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:8:"ek_mukta";a:3:{s:4:"name";s:8:"Ek Mukta";s:8:"variants";s:31:"200,300,regular,500,600,700,800";s:8:"fallback";s:10:"sans-serif";}s:24:"annie_use_your_telescope";a:3:{s:4:"name";s:24:"Annie Use Your Telescope";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"grand_hotel";a:3:{s:4:"name";s:11:"Grand Hotel";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"trocchi";a:3:{s:4:"name";s:7:"Trocchi";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:21:"goudy_bookletter_1911";a:3:{s:4:"name";s:21:"Goudy Bookletter 1911";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:10:"parisienne";a:3:{s:4:"name";s:10:"Parisienne";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:8:"aclonica";a:3:{s:4:"name";s:8:"Aclonica";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"timmana";a:3:{s:4:"name";s:7:"Timmana";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"hanuman";a:3:{s:4:"name";s:7:"Hanuman";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:5:"forum";a:3:{s:4:"name";s:5:"Forum";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"press_start_2p";a:3:{s:4:"name";s:14:"Press Start 2P";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"oranienbaum";a:3:{s:4:"name";s:11:"Oranienbaum";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:11:"sansita_one";a:3:{s:4:"name";s:11:"Sansita One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"unica_one";a:3:{s:4:"name";s:9:"Unica One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"marcellus_sc";a:3:{s:4:"name";s:12:"Marcellus SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:8:"denk_one";a:3:{s:4:"name";s:8:"Denk One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"monoton";a:3:{s:4:"name";s:7:"Monoton";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"corben";a:3:{s:4:"name";s:6:"Corben";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:7:"rosario";a:3:{s:4:"name";s:7:"Rosario";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:17:"cinzel_decorative";a:3:{s:4:"name";s:17:"Cinzel Decorative";s:8:"variants";s:15:"regular,700,900";s:8:"fallback";s:7:"display";}s:10:"schoolbell";a:3:{s:4:"name";s:10:"Schoolbell";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:16:"port_lligat_slab";a:3:{s:4:"name";s:16:"Port Lligat Slab";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:10:"kelly_slab";a:3:{s:4:"name";s:10:"Kelly Slab";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"yesteryear";a:3:{s:4:"name";s:10:"Yesteryear";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"arbutus_slab";a:3:{s:4:"name";s:12:"Arbutus Slab";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:7:"frijole";a:3:{s:4:"name";s:7:"Frijole";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"belleza";a:3:{s:4:"name";s:7:"Belleza";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:4:"alef";a:3:{s:4:"name";s:4:"Alef";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:12:"leckerli_one";a:3:{s:4:"name";s:12:"Leckerli One";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"alegreya_sc";a:3:{s:4:"name";s:11:"Alegreya SC";s:8:"variants";s:42:"regular,italic,700,700italic,900,900italic";s:8:"fallback";s:5:"serif";}s:6:"caudex";a:3:{s:4:"name";s:6:"Caudex";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:12:"merienda_one";a:3:{s:4:"name";s:12:"Merienda One";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:4:"mako";a:3:{s:4:"name";s:4:"Mako";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"cousine";a:3:{s:4:"name";s:7:"Cousine";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:9:"monospace";}s:11:"short_stack";a:3:{s:4:"name";s:11:"Short Stack";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"convergence";a:3:{s:4:"name";s:11:"Convergence";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"inder";a:3:{s:4:"name";s:5:"Inder";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:14:"give_you_glory";a:3:{s:4:"name";s:14:"Give You Glory";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"lustria";a:3:{s:4:"name";s:7:"Lustria";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:6:"gruppo";a:3:{s:4:"name";s:6:"Gruppo";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"slackey";a:3:{s:4:"name";s:7:"Slackey";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"tenor_sans";a:3:{s:4:"name";s:10:"Tenor Sans";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"capriola";a:3:{s:4:"name";s:8:"Capriola";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:11:"metrophobic";a:3:{s:4:"name";s:11:"Metrophobic";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"graduate";a:3:{s:4:"name";s:8:"Graduate";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:15:"racing_sans_one";a:3:{s:4:"name";s:15:"Racing Sans One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"prosto_one";a:3:{s:4:"name";s:10:"Prosto One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:19:"sue_ellen_francisco";a:3:{s:4:"name";s:19:"Sue Ellen Francisco";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"alike";a:3:{s:4:"name";s:5:"Alike";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"duru_sans";a:3:{s:4:"name";s:9:"Duru Sans";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:10:"lilita_one";a:3:{s:4:"name";s:10:"Lilita One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"italianno";a:3:{s:4:"name";s:9:"Italianno";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"unkempt";a:3:{s:4:"name";s:7:"Unkempt";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:10:"carter_one";a:3:{s:4:"name";s:10:"Carter One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"nova_square";a:3:{s:4:"name";s:11:"Nova Square";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"baumans";a:3:{s:4:"name";s:7:"Baumans";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"poller_one";a:3:{s:4:"name";s:10:"Poller One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"lemon";a:3:{s:4:"name";s:5:"Lemon";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:19:"petit_formal_script";a:3:{s:4:"name";s:19:"Petit Formal Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:10:"engagement";a:3:{s:4:"name";s:10:"Engagement";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:23:"just_me_again_down_here";a:3:{s:4:"name";s:23:"Just Me Again Down Here";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:9:"creepster";a:3:{s:4:"name";s:9:"Creepster";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"fenix";a:3:{s:4:"name";s:5:"Fenix";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:6:"strait";a:3:{s:4:"name";s:6:"Strait";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"sarala";a:3:{s:4:"name";s:6:"Sarala";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:14:"pragati_narrow";a:3:{s:4:"name";s:14:"Pragati Narrow";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:13:"gilda_display";a:3:{s:4:"name";s:13:"Gilda Display";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"tauri";a:3:{s:4:"name";s:5:"Tauri";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:14:"mr_de_haviland";a:3:{s:4:"name";s:14:"Mr De Haviland";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"skranji";a:3:{s:4:"name";s:7:"Skranji";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:18:"the_girl_next_door";a:3:{s:4:"name";s:18:"The Girl Next Door";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"lateef";a:3:{s:4:"name";s:6:"Lateef";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"delius";a:3:{s:4:"name";s:6:"Delius";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:14:"londrina_solid";a:3:{s:4:"name";s:14:"Londrina Solid";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"bowlby_one_sc";a:3:{s:4:"name";s:13:"Bowlby One SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"imprima";a:3:{s:4:"name";s:7:"Imprima";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"quando";a:3:{s:4:"name";s:6:"Quando";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:6:"rufina";a:3:{s:4:"name";s:6:"Rufina";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:5:"allan";a:3:{s:4:"name";s:5:"Allan";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:7:"anaheim";a:3:{s:4:"name";s:7:"Anaheim";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"crushed";a:3:{s:4:"name";s:7:"Crushed";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"judson";a:3:{s:4:"name";s:6:"Judson";s:8:"variants";s:18:"regular,italic,700";s:8:"fallback";s:5:"serif";}s:11:"oxygen_mono";a:3:{s:4:"name";s:11:"Oxygen Mono";s:8:"variants";s:7:"regular";s:8:"fallback";s:9:"monospace";}s:7:"knewave";a:3:{s:4:"name";s:7:"Knewave";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"kotta_one";a:3:{s:4:"name";s:9:"Kotta One";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:13:"hind_siliguri";a:3:{s:4:"name";s:13:"Hind Siliguri";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:7:"brawler";a:3:{s:4:"name";s:7:"Brawler";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:8:"wire_one";a:3:{s:4:"name";s:8:"Wire One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:17:"loved_by_the_king";a:3:{s:4:"name";s:17:"Loved by the King";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"megrim";a:3:{s:4:"name";s:6:"Megrim";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"andika";a:3:{s:4:"name";s:6:"Andika";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:15:"im_fell_dw_pica";a:3:{s:4:"name";s:15:"IM Fell DW Pica";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:3:"ovo";a:3:{s:4:"name";s:3:"Ovo";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:12:"gravitas_one";a:3:{s:4:"name";s:12:"Gravitas One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:15:"la_belle_aurore";a:3:{s:4:"name";s:15:"La Belle Aurore";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:10:"slabo_13px";a:3:{s:4:"name";s:10:"Slabo 13px";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:7:"buenard";a:3:{s:4:"name";s:7:"Buenard";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:12:"headland_one";a:3:{s:4:"name";s:12:"Headland One";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:22:"oleo_script_swash_caps";a:3:{s:4:"name";s:22:"Oleo Script Swash Caps";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:7:"oregano";a:3:{s:4:"name";s:7:"Oregano";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:7:"display";}s:6:"andada";a:3:{s:4:"name";s:6:"Andada";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"fjord_one";a:3:{s:4:"name";s:9:"Fjord One";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:12:"fanwood_text";a:3:{s:4:"name";s:12:"Fanwood Text";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:11:"cutive_mono";a:3:{s:4:"name";s:11:"Cutive Mono";s:8:"variants";s:7:"regular";s:8:"fallback";s:9:"monospace";}s:9:"rationale";a:3:{s:4:"name";s:9:"Rationale";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"pompiere";a:3:{s:4:"name";s:8:"Pompiere";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"vt323";a:3:{s:4:"name";s:5:"VT323";s:8:"variants";s:7:"regular";s:8:"fallback";s:9:"monospace";}s:18:"averia_serif_libre";a:3:{s:4:"name";s:18:"Averia Serif Libre";s:8:"variants";s:42:"300,300italic,regular,italic,700,700italic";s:8:"fallback";s:7:"display";}s:20:"herr_von_muellerhoff";a:3:{s:4:"name";s:20:"Herr Von Muellerhoff";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:13:"anonymous_pro";a:3:{s:4:"name";s:13:"Anonymous Pro";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:9:"monospace";}s:18:"unifrakturmaguntia";a:3:{s:4:"name";s:18:"UnifrakturMaguntia";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"bentham";a:3:{s:4:"name";s:7:"Bentham";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:10:"yeseva_one";a:3:{s:4:"name";s:10:"Yeseva One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"gafata";a:3:{s:4:"name";s:6:"Gafata";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:12:"happy_monkey";a:3:{s:4:"name";s:12:"Happy Monkey";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:18:"im_fell_english_sc";a:3:{s:4:"name";s:18:"IM Fell English SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:17:"averia_sans_libre";a:3:{s:4:"name";s:17:"Averia Sans Libre";s:8:"variants";s:42:"300,300italic,regular,italic,700,700italic";s:8:"fallback";s:7:"display";}s:15:"caesar_dressing";a:3:{s:4:"name";s:15:"Caesar Dressing";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:4:"mate";a:3:{s:4:"name";s:4:"Mate";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:6:"kranky";a:3:{s:4:"name";s:6:"Kranky";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:21:"love_ya_like_a_sister";a:3:{s:4:"name";s:21:"Love Ya Like A Sister";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"tienne";a:3:{s:4:"name";s:6:"Tienne";s:8:"variants";s:15:"regular,700,900";s:8:"fallback";s:5:"serif";}s:22:"mountains_of_christmas";a:3:{s:4:"name";s:22:"Mountains of Christmas";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:7:"orienta";a:3:{s:4:"name";s:7:"Orienta";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"aladin";a:3:{s:4:"name";s:6:"Aladin";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:14:"seaweed_script";a:3:{s:4:"name";s:14:"Seaweed Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"belgrano";a:3:{s:4:"name";s:8:"Belgrano";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:7:"biryani";a:3:{s:4:"name";s:7:"Biryani";s:8:"variants";s:31:"200,300,regular,600,700,800,900";s:8:"fallback";s:10:"sans-serif";}s:9:"simonetta";a:3:{s:4:"name";s:9:"Simonetta";s:8:"variants";s:28:"regular,italic,900,900italic";s:8:"fallback";s:7:"display";}s:15:"stardos_stencil";a:3:{s:4:"name";s:15:"Stardos Stencil";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:15:"lily_script_one";a:3:{s:4:"name";s:15:"Lily Script One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"norican";a:3:{s:4:"name";s:7:"Norican";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:9:"balthazar";a:3:{s:4:"name";s:9:"Balthazar";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"gfs_didot";a:3:{s:4:"name";s:9:"GFS Didot";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:15:"euphoria_script";a:3:{s:4:"name";s:15:"Euphoria Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:9:"englebert";a:3:{s:4:"name";s:9:"Englebert";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"qwigley";a:3:{s:4:"name";s:7:"Qwigley";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:16:"over_the_rainbow";a:3:{s:4:"name";s:16:"Over the Rainbow";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"inika";a:3:{s:4:"name";s:5:"Inika";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:4:"itim";a:3:{s:4:"name";s:4:"Itim";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"sofia";a:3:{s:4:"name";s:5:"Sofia";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"griffy";a:3:{s:4:"name";s:6:"Griffy";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:15:"im_fell_english";a:3:{s:4:"name";s:15:"IM Fell English";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:5:"khmer";a:3:{s:4:"name";s:5:"Khmer";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"henny_penny";a:3:{s:4:"name";s:11:"Henny Penny";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"bowlby_one";a:3:{s:4:"name";s:10:"Bowlby One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"quintessential";a:3:{s:4:"name";s:14:"Quintessential";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"podkova";a:3:{s:4:"name";s:7:"Podkova";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:18:"chau_philomene_one";a:3:{s:4:"name";s:18:"Chau Philomene One";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:10:"sans-serif";}s:8:"mr_dafoe";a:3:{s:4:"name";s:8:"Mr Dafoe";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:8:"arizonia";a:3:{s:4:"name";s:8:"Arizonia";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"dorsa";a:3:{s:4:"name";s:5:"Dorsa";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:17:"carrois_gothic_sc";a:3:{s:4:"name";s:17:"Carrois Gothic SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"salsa";a:3:{s:4:"name";s:5:"Salsa";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"shanti";a:3:{s:4:"name";s:6:"Shanti";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:3:"geo";a:3:{s:4:"name";s:3:"Geo";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:10:"sans-serif";}s:12:"caveat_brush";a:3:{s:4:"name";s:12:"Caveat Brush";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"karma";a:3:{s:4:"name";s:5:"Karma";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:5:"serif";}s:13:"expletus_sans";a:3:{s:4:"name";s:13:"Expletus Sans";s:8:"variants";s:56:"regular,italic,500,500italic,600,600italic,700,700italic";s:8:"fallback";s:7:"display";}s:10:"share_tech";a:3:{s:4:"name";s:10:"Share Tech";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"meddon";a:3:{s:4:"name";s:6:"Meddon";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:8:"gabriela";a:3:{s:4:"name";s:8:"Gabriela";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"fira_mono";a:3:{s:4:"name";s:9:"Fira Mono";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:9:"monospace";}s:4:"poly";a:3:{s:4:"name";s:4:"Poly";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:13:"hind_vadodara";a:3:{s:4:"name";s:13:"Hind Vadodara";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:7:"mate_sc";a:3:{s:4:"name";s:7:"Mate SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"stalemate";a:3:{s:4:"name";s:9:"Stalemate";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"federo";a:3:{s:4:"name";s:6:"Federo";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"italiana";a:3:{s:4:"name";s:8:"Italiana";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:11:"concert_one";a:3:{s:4:"name";s:11:"Concert One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"fondamento";a:3:{s:4:"name";s:10:"Fondamento";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:11:"handwriting";}s:11:"life_savers";a:3:{s:4:"name";s:11:"Life Savers";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:9:"oldenburg";a:3:{s:4:"name";s:9:"Oldenburg";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:16:"bilbo_swash_caps";a:3:{s:4:"name";s:16:"Bilbo Swash Caps";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"galindo";a:3:{s:4:"name";s:7:"Galindo";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:3:"ntr";a:3:{s:4:"name";s:3:"NTR";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:17:"delius_swash_caps";a:3:{s:4:"name";s:17:"Delius Swash Caps";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"sniglet";a:3:{s:4:"name";s:7:"Sniglet";s:8:"variants";s:11:"regular,800";s:8:"fallback";s:7:"display";}s:6:"kristi";a:3:{s:4:"name";s:6:"Kristi";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:13:"maiden_orange";a:3:{s:4:"name";s:13:"Maiden Orange";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"nova_mono";a:3:{s:4:"name";s:9:"Nova Mono";s:8:"variants";s:7:"regular";s:8:"fallback";s:9:"monospace";}s:4:"unna";a:3:{s:4:"name";s:4:"Unna";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"voces";a:3:{s:4:"name";s:5:"Voces";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"martel";a:3:{s:4:"name";s:6:"Martel";s:8:"variants";s:31:"200,300,regular,600,700,800,900";s:8:"fallback";s:5:"serif";}s:8:"codystar";a:3:{s:4:"name";s:8:"Codystar";s:8:"variants";s:11:"300,regular";s:8:"fallback";s:7:"display";}s:15:"holtwood_one_sc";a:3:{s:4:"name";s:15:"Holtwood One SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"cambo";a:3:{s:4:"name";s:5:"Cambo";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:14:"clicker_script";a:3:{s:4:"name";s:14:"Clicker Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"coda_caption";a:3:{s:4:"name";s:12:"Coda Caption";s:8:"variants";s:3:"800";s:8:"fallback";s:10:"sans-serif";}s:8:"kite_one";a:3:{s:4:"name";s:8:"Kite One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"mallanna";a:3:{s:4:"name";s:8:"Mallanna";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:12:"raleway_dots";a:3:{s:4:"name";s:12:"Raleway Dots";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:15:"patrick_hand_sc";a:3:{s:4:"name";s:15:"Patrick Hand SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:18:"cedarville_cursive";a:3:{s:4:"name";s:18:"Cedarville Cursive";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:4:"sail";a:3:{s:4:"name";s:4:"Sail";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"metamorphous";a:3:{s:4:"name";s:12:"Metamorphous";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"nosifer";a:3:{s:4:"name";s:7:"Nosifer";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"cherry_swash";a:3:{s:4:"name";s:12:"Cherry Swash";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:10:"tulpen_one";a:3:{s:4:"name";s:10:"Tulpen One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:19:"im_fell_double_pica";a:3:{s:4:"name";s:19:"IM Fell Double Pica";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:20:"dawning_of_a_new_day";a:3:{s:4:"name";s:20:"Dawning of a New Day";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"esteban";a:3:{s:4:"name";s:7:"Esteban";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"amethysta";a:3:{s:4:"name";s:9:"Amethysta";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"condiment";a:3:{s:4:"name";s:9:"Condiment";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"rouge_script";a:3:{s:4:"name";s:12:"Rouge Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:8:"flamenco";a:3:{s:4:"name";s:8:"Flamenco";s:8:"variants";s:11:"300,regular";s:8:"fallback";s:7:"display";}s:9:"shojumaru";a:3:{s:4:"name";s:9:"Shojumaru";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"ledger";a:3:{s:4:"name";s:6:"Ledger";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:11:"cantora_one";a:3:{s:4:"name";s:11:"Cantora One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:11:"yantramanav";a:3:{s:4:"name";s:11:"Yantramanav";s:8:"variants";s:27:"100,300,regular,500,700,900";s:8:"fallback";s:10:"sans-serif";}s:15:"aguafina_script";a:3:{s:4:"name";s:15:"Aguafina Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:14:"ruslan_display";a:3:{s:4:"name";s:14:"Ruslan Display";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"rammetto_one";a:3:{s:4:"name";s:12:"Rammetto One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"rubik_one";a:3:{s:4:"name";s:9:"Rubik One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"titan_one";a:3:{s:4:"name";s:9:"Titan One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"zeyada";a:3:{s:4:"name";s:6:"Zeyada";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:15:"share_tech_mono";a:3:{s:4:"name";s:15:"Share Tech Mono";s:8:"variants";s:7:"regular";s:8:"fallback";s:9:"monospace";}s:7:"milonga";a:3:{s:4:"name";s:7:"Milonga";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"vast_shadow";a:3:{s:4:"name";s:11:"Vast Shadow";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"sancreek";a:3:{s:4:"name";s:8:"Sancreek";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"junge";a:3:{s:4:"name";s:5:"Junge";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:11:"trade_winds";a:3:{s:4:"name";s:11:"Trade Winds";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:3:"rye";a:3:{s:4:"name";s:3:"Rye";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:18:"swanky_and_moo_moo";a:3:{s:4:"name";s:18:"Swanky and Moo Moo";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:9:"palanquin";a:3:{s:4:"name";s:9:"Palanquin";s:8:"variants";s:31:"100,200,300,regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:8:"amarante";a:3:{s:4:"name";s:8:"Amarante";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:21:"stint_ultra_condensed";a:3:{s:4:"name";s:21:"Stint Ultra Condensed";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"iceland";a:3:{s:4:"name";s:7:"Iceland";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"delius_unicase";a:3:{s:4:"name";s:14:"Delius Unicase";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:11:"handwriting";}s:11:"martel_sans";a:3:{s:4:"name";s:11:"Martel Sans";s:8:"variants";s:31:"200,300,regular,600,700,800,900";s:8:"fallback";s:10:"sans-serif";}s:10:"medula_one";a:3:{s:4:"name";s:10:"Medula One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"gurajada";a:3:{s:4:"name";s:8:"Gurajada";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:10:"ramabhadra";a:3:{s:4:"name";s:10:"Ramabhadra";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:8:"monofett";a:3:{s:4:"name";s:8:"Monofett";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"mclaren";a:3:{s:4:"name";s:7:"McLaren";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:22:"im_fell_double_pica_sc";a:3:{s:4:"name";s:22:"IM Fell Double Pica SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:20:"im_fell_french_canon";a:3:{s:4:"name";s:20:"IM Fell French Canon";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:10:"butcherman";a:3:{s:4:"name";s:10:"Butcherman";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"artifika";a:3:{s:4:"name";s:8:"Artifika";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:6:"angkor";a:3:{s:4:"name";s:6:"Angkor";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"dynalight";a:3:{s:4:"name";s:9:"Dynalight";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"overlock_sc";a:3:{s:4:"name";s:11:"Overlock SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"kavoon";a:3:{s:4:"name";s:6:"Kavoon";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"kurale";a:3:{s:4:"name";s:6:"Kurale";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:11:"donegal_one";a:3:{s:4:"name";s:11:"Donegal One";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"redressed";a:3:{s:4:"name";s:9:"Redressed";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"paprika";a:3:{s:4:"name";s:7:"Paprika";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"stoke";a:3:{s:4:"name";s:5:"Stoke";s:8:"variants";s:11:"300,regular";s:8:"fallback";s:5:"serif";}s:6:"nokora";a:3:{s:4:"name";s:6:"Nokora";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:8:"wallpoet";a:3:{s:4:"name";s:8:"Wallpoet";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"wendy_one";a:3:{s:4:"name";s:9:"Wendy One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"sarina";a:3:{s:4:"name";s:6:"Sarina";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"jolly_lodger";a:3:{s:4:"name";s:12:"Jolly Lodger";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"nova_round";a:3:{s:4:"name";s:10:"Nova Round";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"ruthie";a:3:{s:4:"name";s:6:"Ruthie";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:10:"cagliostro";a:3:{s:4:"name";s:10:"Cagliostro";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:18:"im_fell_dw_pica_sc";a:3:{s:4:"name";s:18:"IM Fell DW Pica SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:16:"jacques_francois";a:3:{s:4:"name";s:16:"Jacques Francois";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:8:"rosarivo";a:3:{s:4:"name";s:8:"Rosarivo";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:10:"pirata_one";a:3:{s:4:"name";s:10:"Pirata One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"prociono";a:3:{s:4:"name";s:8:"Prociono";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:12:"averia_libre";a:3:{s:4:"name";s:12:"Averia Libre";s:8:"variants";s:42:"300,300italic,regular,italic,700,700italic";s:8:"fallback";s:7:"display";}s:23:"im_fell_french_canon_sc";a:3:{s:4:"name";s:23:"IM Fell French Canon SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"sunshiney";a:3:{s:4:"name";s:9:"Sunshiney";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:20:"montserrat_subrayada";a:3:{s:4:"name";s:20:"Montserrat Subrayada";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:6:"habibi";a:3:{s:4:"name";s:6:"Habibi";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:20:"stint_ultra_expanded";a:3:{s:4:"name";s:20:"Stint Ultra Expanded";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:23:"im_fell_great_primer_sc";a:3:{s:4:"name";s:23:"IM Fell Great Primer SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:9:"krona_one";a:3:{s:4:"name";s:9:"Krona One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"ruluko";a:3:{s:4:"name";s:6:"Ruluko";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"piedra";a:3:{s:4:"name";s:6:"Piedra";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"text_me_one";a:3:{s:4:"name";s:11:"Text Me One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:4:"asul";a:3:{s:4:"name";s:4:"Asul";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:12:"scheherazade";a:3:{s:4:"name";s:12:"Scheherazade";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:6:"numans";a:3:{s:4:"name";s:6:"Numans";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:20:"im_fell_great_primer";a:3:{s:4:"name";s:20:"IM Fell Great Primer";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:10:"sonsie_one";a:3:{s:4:"name";s:10:"Sonsie One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"rubik_mono_one";a:3:{s:4:"name";s:14:"Rubik Mono One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:11:"linden_hill";a:3:{s:4:"name";s:11:"Linden Hill";s:8:"variants";s:14:"regular,italic";s:8:"fallback";s:5:"serif";}s:16:"port_lligat_sans";a:3:{s:4:"name";s:16:"Port Lligat Sans";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"vibur";a:3:{s:4:"name";s:5:"Vibur";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"irish_grover";a:3:{s:4:"name";s:12:"Irish Grover";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:4:"buda";a:3:{s:4:"name";s:4:"Buda";s:8:"variants";s:3:"300";s:8:"fallback";s:7:"display";}s:7:"offside";a:3:{s:4:"name";s:7:"Offside";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"nova_slim";a:3:{s:4:"name";s:9:"Nova Slim";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"snippet";a:3:{s:4:"name";s:7:"Snippet";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:19:"mrs_saint_delafield";a:3:{s:4:"name";s:19:"Mrs Saint Delafield";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"caveat";a:3:{s:4:"name";s:6:"Caveat";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:11:"handwriting";}s:5:"bilbo";a:3:{s:4:"name";s:5:"Bilbo";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:13:"della_respira";a:3:{s:4:"name";s:13:"Della Respira";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:13:"glass_antiqua";a:3:{s:4:"name";s:13:"Glass Antiqua";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"ribeye";a:3:{s:4:"name";s:6:"Ribeye";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"battambang";a:3:{s:4:"name";s:10:"Battambang";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:10:"new_rocker";a:3:{s:4:"name";s:10:"New Rocker";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"wellfleet";a:3:{s:4:"name";s:9:"Wellfleet";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"smythe";a:3:{s:4:"name";s:6:"Smythe";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"mystery_quest";a:3:{s:4:"name";s:13:"Mystery Quest";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"elsie";a:3:{s:4:"name";s:5:"Elsie";s:8:"variants";s:11:"regular,900";s:8:"fallback";s:7:"display";}s:15:"gfs_neohellenic";a:3:{s:4:"name";s:15:"GFS Neohellenic";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:10:"sans-serif";}s:7:"trochut";a:3:{s:4:"name";s:7:"Trochut";s:8:"variants";s:18:"regular,italic,700";s:8:"fallback";s:7:"display";}s:10:"keania_one";a:3:{s:4:"name";s:10:"Keania One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"league_script";a:3:{s:4:"name";s:13:"League Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"suranna";a:3:{s:4:"name";s:7:"Suranna";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:6:"chango";a:3:{s:4:"name";s:6:"Chango";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"peralta";a:3:{s:4:"name";s:7:"Peralta";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"antic_didone";a:3:{s:4:"name";s:12:"Antic Didone";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:13:"alike_angular";a:3:{s:4:"name";s:13:"Alike Angular";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:11:"dr_sugiyama";a:3:{s:4:"name";s:11:"Dr Sugiyama";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:5:"kenia";a:3:{s:4:"name";s:5:"Kenia";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"content";a:3:{s:4:"name";s:7:"Content";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:5:"julee";a:3:{s:4:"name";s:5:"Julee";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:13:"snowburst_one";a:3:{s:4:"name";s:13:"Snowburst One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"medievalsharp";a:3:{s:4:"name";s:13:"MedievalSharp";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"unifrakturcook";a:3:{s:4:"name";s:14:"UnifrakturCook";s:8:"variants";s:3:"700";s:8:"fallback";s:7:"display";}s:11:"suwannaphum";a:3:{s:4:"name";s:11:"Suwannaphum";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"bigshot_one";a:3:{s:4:"name";s:11:"Bigshot One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"bubbler_one";a:3:{s:4:"name";s:11:"Bubbler One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"miniver";a:3:{s:4:"name";s:7:"Miniver";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"sumana";a:3:{s:4:"name";s:6:"Sumana";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:8:"joti_one";a:3:{s:4:"name";s:8:"Joti One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"akronim";a:3:{s:4:"name";s:7:"Akronim";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"lovers_quarrel";a:3:{s:4:"name";s:14:"Lovers Quarrel";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"germania_one";a:3:{s:4:"name";s:12:"Germania One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:23:"jacques_francois_shadow";a:3:{s:4:"name";s:23:"Jacques Francois Shadow";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"trykker";a:3:{s:4:"name";s:7:"Trykker";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:7:"astloch";a:3:{s:4:"name";s:7:"Astloch";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:10:"autour_one";a:3:{s:4:"name";s:10:"Autour One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:16:"miltonian_tattoo";a:3:{s:4:"name";s:16:"Miltonian Tattoo";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"combo";a:3:{s:4:"name";s:5:"Combo";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"nova_flat";a:3:{s:4:"name";s:9:"Nova Flat";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:16:"elsie_swash_caps";a:3:{s:4:"name";s:16:"Elsie Swash Caps";s:8:"variants";s:11:"regular,900";s:8:"fallback";s:7:"display";}s:6:"warnes";a:3:{s:4:"name";s:6:"Warnes";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"iceberg";a:3:{s:4:"name";s:7:"Iceberg";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"ribeye_marrow";a:3:{s:4:"name";s:13:"Ribeye Marrow";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"galdeano";a:3:{s:4:"name";s:8:"Galdeano";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:15:"londrina_shadow";a:3:{s:4:"name";s:15:"Londrina Shadow";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"ranchers";a:3:{s:4:"name";s:8:"Ranchers";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"croissant_one";a:3:{s:4:"name";s:13:"Croissant One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"spicy_rice";a:3:{s:4:"name";s:10:"Spicy Rice";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:19:"averia_gruesa_libre";a:3:{s:4:"name";s:19:"Averia Gruesa Libre";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"modern_antiqua";a:3:{s:4:"name";s:14:"Modern Antiqua";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:15:"londrina_sketch";a:3:{s:4:"name";s:15:"Londrina Sketch";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"fresca";a:3:{s:4:"name";s:6:"Fresca";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"petrona";a:3:{s:4:"name";s:7:"Petrona";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"jaldi";a:3:{s:4:"name";s:5:"Jaldi";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:10:"eagle_lake";a:3:{s:4:"name";s:10:"Eagle Lake";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"passero_one";a:3:{s:4:"name";s:11:"Passero One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"sofadi_one";a:3:{s:4:"name";s:10:"Sofadi One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"atomic_age";a:3:{s:4:"name";s:10:"Atomic Age";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"freehand";a:3:{s:4:"name";s:8:"Freehand";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"asset";a:3:{s:4:"name";s:5:"Asset";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:4:"sura";a:3:{s:4:"name";s:4:"Sura";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:20:"monsieur_la_doulaise";a:3:{s:4:"name";s:20:"Monsieur La Doulaise";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:15:"original_surfer";a:3:{s:4:"name";s:15:"Original Surfer";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"fascinate";a:3:{s:4:"name";s:9:"Fascinate";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"purple_purse";a:3:{s:4:"name";s:12:"Purple Purse";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"margarine";a:3:{s:4:"name";s:9:"Margarine";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"sarpanch";a:3:{s:4:"name";s:8:"Sarpanch";s:8:"variants";s:27:"regular,500,600,700,800,900";s:8:"fallback";s:10:"sans-serif";}s:9:"diplomata";a:3:{s:4:"name";s:9:"Diplomata";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"lancelot";a:3:{s:4:"name";s:8:"Lancelot";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"gorditas";a:3:{s:4:"name";s:8:"Gorditas";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:8:"underdog";a:3:{s:4:"name";s:8:"Underdog";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"faster_one";a:3:{s:4:"name";s:10:"Faster One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"montaga";a:3:{s:4:"name";s:7:"Montaga";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:8:"almendra";a:3:{s:4:"name";s:8:"Almendra";s:8:"variants";s:28:"regular,italic,700,700italic";s:8:"fallback";s:5:"serif";}s:9:"rozha_one";a:3:{s:4:"name";s:9:"Rozha One";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:10:"devonshire";a:3:{s:4:"name";s:10:"Devonshire";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:12:"emilys_candy";a:3:{s:4:"name";s:12:"Emilys Candy";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"kdam_thmor";a:3:{s:4:"name";s:10:"Kdam Thmor";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"geostar_fill";a:3:{s:4:"name";s:12:"Geostar Fill";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"miltonian";a:3:{s:4:"name";s:9:"Miltonian";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"dekko";a:3:{s:4:"name";s:5:"Dekko";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"aubrey";a:3:{s:4:"name";s:6:"Aubrey";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"vampiro_one";a:3:{s:4:"name";s:11:"Vampiro One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"mandali";a:3:{s:4:"name";s:7:"Mandali";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:11:"metal_mania";a:3:{s:4:"name";s:11:"Metal Mania";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:16:"fascinate_inline";a:3:{s:4:"name";s:16:"Fascinate Inline";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"palanquin_dark";a:3:{s:4:"name";s:14:"Palanquin Dark";s:8:"variants";s:19:"regular,500,600,700";s:8:"fallback";s:10:"sans-serif";}s:5:"laila";a:3:{s:4:"name";s:5:"Laila";s:8:"variants";s:23:"300,regular,500,600,700";s:8:"fallback";s:5:"serif";}s:6:"smokum";a:3:{s:4:"name";s:6:"Smokum";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"mrs_sheppards";a:3:{s:4:"name";s:13:"Mrs Sheppards";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:11:"meie_script";a:3:{s:4:"name";s:11:"Meie Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:10:"goblin_one";a:3:{s:4:"name";s:10:"Goblin One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"ewert";a:3:{s:4:"name";s:5:"Ewert";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"seymour_one";a:3:{s:4:"name";s:11:"Seymour One";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:5:"bokor";a:3:{s:4:"name";s:5:"Bokor";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"chicle";a:3:{s:4:"name";s:6:"Chicle";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"butterfly_kids";a:3:{s:4:"name";s:14:"Butterfly Kids";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:8:"nova_cut";a:3:{s:4:"name";s:8:"Nova Cut";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"nova_script";a:3:{s:4:"name";s:11:"Nova Script";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"felipa";a:3:{s:4:"name";s:6:"Felipa";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:8:"siemreap";a:3:{s:4:"name";s:8:"Siemreap";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"revalia";a:3:{s:4:"name";s:7:"Revalia";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"princess_sofia";a:3:{s:4:"name";s:14:"Princess Sofia";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:18:"macondo_swash_caps";a:3:{s:4:"name";s:18:"Macondo Swash Caps";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"romanesco";a:3:{s:4:"name";s:9:"Romanesco";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:8:"federant";a:3:{s:4:"name";s:8:"Federant";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"rum_raisin";a:3:{s:4:"name";s:10:"Rum Raisin";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"nova_oval";a:3:{s:4:"name";s:9:"Nova Oval";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"spirax";a:3:{s:4:"name";s:6:"Spirax";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"molle";a:3:{s:4:"name";s:5:"Molle";s:8:"variants";s:6:"italic";s:8:"fallback";s:11:"handwriting";}s:16:"almendra_display";a:3:{s:4:"name";s:16:"Almendra Display";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"ranga";a:3:{s:4:"name";s:5:"Ranga";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:7:"display";}s:7:"dangrek";a:3:{s:4:"name";s:7:"Dangrek";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"geostar";a:3:{s:4:"name";s:7:"Geostar";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"marko_one";a:3:{s:4:"name";s:9:"Marko One";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:8:"chonburi";a:3:{s:4:"name";s:8:"Chonburi";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:16:"londrina_outline";a:3:{s:4:"name";s:16:"Londrina Outline";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"dhurjati";a:3:{s:4:"name";s:8:"Dhurjati";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:7:"plaster";a:3:{s:4:"name";s:7:"Plaster";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"diplomata_sc";a:3:{s:4:"name";s:12:"Diplomata SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:4:"moul";a:3:{s:4:"name";s:4:"Moul";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:16:"supermercado_one";a:3:{s:4:"name";s:16:"Supermercado One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"chela_one";a:3:{s:4:"name";s:9:"Chela One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"eater";a:3:{s:4:"name";s:5:"Eater";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"vesper_libre";a:3:{s:4:"name";s:12:"Vesper Libre";s:8:"variants";s:19:"regular,500,700,900";s:8:"fallback";s:5:"serif";}s:14:"uncial_antiqua";a:3:{s:4:"name";s:14:"Uncial Antiqua";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"ramaraja";a:3:{s:4:"name";s:8:"Ramaraja";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"amita";a:3:{s:4:"name";s:5:"Amita";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:11:"handwriting";}s:5:"bayon";a:3:{s:4:"name";s:5:"Bayon";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"miss_fajardose";a:3:{s:4:"name";s:14:"Miss Fajardose";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:13:"rhodium_libre";a:3:{s:4:"name";s:13:"Rhodium Libre";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:13:"bigelow_rules";a:3:{s:4:"name";s:13:"Bigelow Rules";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"macondo";a:3:{s:4:"name";s:7:"Macondo";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:4:"arya";a:3:{s:4:"name";s:4:"Arya";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:10:"sans-serif";}s:13:"stalinist_one";a:3:{s:4:"name";s:13:"Stalinist One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"metal";a:3:{s:4:"name";s:5:"Metal";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:9:"sevillana";a:3:{s:4:"name";s:9:"Sevillana";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"arbutus";a:3:{s:4:"name";s:7:"Arbutus";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:13:"sirin_stencil";a:3:{s:4:"name";s:13:"Sirin Stencil";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"bonbon";a:3:{s:4:"name";s:6:"Bonbon";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"koulen";a:3:{s:4:"name";s:6:"Koulen";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"taprom";a:3:{s:4:"name";s:6:"Taprom";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:10:"mr_bedfort";a:3:{s:4:"name";s:10:"Mr Bedfort";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:6:"gidugu";a:3:{s:4:"name";s:6:"Gidugu";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:6:"risque";a:3:{s:4:"name";s:6:"Risque";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:6:"chenla";a:3:{s:4:"name";s:6:"Chenla";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"almendra_sc";a:3:{s:4:"name";s:11:"Almendra SC";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:7:"flavors";a:3:{s:4:"name";s:7:"Flavors";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"emblema_one";a:3:{s:4:"name";s:11:"Emblema One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:14:"odor_mean_chey";a:3:{s:4:"name";s:14:"Odor Mean Chey";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"ruge_boogie";a:3:{s:4:"name";s:11:"Ruge Boogie";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:14:"jim_nightshade";a:3:{s:4:"name";s:14:"Jim Nightshade";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:7:"tillana";a:3:{s:4:"name";s:7:"Tillana";s:8:"variants";s:23:"regular,500,600,700,800";s:8:"fallback";s:11:"handwriting";}s:9:"erica_one";a:3:{s:4:"name";s:9:"Erica One";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:11:"preahvihear";a:3:{s:4:"name";s:11:"Preahvihear";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:18:"tenali_ramakrishna";a:3:{s:4:"name";s:18:"Tenali Ramakrishna";s:8:"variants";s:7:"regular";s:8:"fallback";s:10:"sans-serif";}s:9:"kantumruy";a:3:{s:4:"name";s:9:"Kantumruy";s:8:"variants";s:15:"300,regular,700";s:8:"fallback";s:10:"sans-serif";}s:9:"suravaram";a:3:{s:4:"name";s:9:"Suravaram";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:11:"lakki_reddy";a:3:{s:4:"name";s:11:"Lakki Reddy";s:8:"variants";s:7:"regular";s:8:"fallback";s:11:"handwriting";}s:20:"sree_krushnadevaraya";a:3:{s:4:"name";s:20:"Sree Krushnadevaraya";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:6:"unlock";a:3:{s:4:"name";s:6:"Unlock";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"hanalei_fill";a:3:{s:4:"name";s:12:"Hanalei Fill";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:8:"fasthand";a:3:{s:4:"name";s:8:"Fasthand";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:8:"moulpali";a:3:{s:4:"name";s:8:"Moulpali";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:7:"sahitya";a:3:{s:4:"name";s:7:"Sahitya";s:8:"variants";s:11:"regular,700";s:8:"fallback";s:5:"serif";}s:7:"hanalei";a:3:{s:4:"name";s:7:"Hanalei";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:5:"eczar";a:3:{s:4:"name";s:5:"Eczar";s:8:"variants";s:23:"regular,500,600,700,800";s:8:"fallback";s:5:"serif";}s:14:"inknut_antiqua";a:3:{s:4:"name";s:14:"Inknut Antiqua";s:8:"variants";s:31:"300,regular,500,600,700,800,900";s:8:"fallback";s:5:"serif";}s:7:"fruktur";a:3:{s:4:"name";s:7:"Fruktur";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:4:"asar";a:3:{s:4:"name";s:4:"Asar";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:7:"peddana";a:3:{s:4:"name";s:7:"Peddana";s:8:"variants";s:7:"regular";s:8:"fallback";s:5:"serif";}s:5:"modak";a:3:{s:4:"name";s:5:"Modak";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}s:12:"ravi_prakash";a:3:{s:4:"name";s:12:"Ravi Prakash";s:8:"variants";s:7:"regular";s:8:"fallback";s:7:"display";}}');
	 
	return $google_fonts_list;

  } // load_google_fonts

  static function getUserIP() {
		if( array_key_exists('HTTP_X_FORWARDED_FOR', $_SERVER) && !empty($_SERVER['HTTP_X_FORWARDED_FOR']) ) {
			if (strpos($_SERVER['HTTP_X_FORWARDED_FOR'], ',')>0) {
				$addr = explode(",",$_SERVER['HTTP_X_FORWARDED_FOR']);
				return trim($addr[0]);
			} else {
				return $_SERVER['HTTP_X_FORWARDED_FOR'];
			}
		}
		else if(!empty($_SERVER['REMOTE_ADDR'])){
			return $_SERVER['REMOTE_ADDR'];
		} else {
			return 'unknown.ip';	
		}
	}


  // setup everything when plugin is  activated
  static function activate() {
    require_once(ABSPATH . 'wp-admin/includes/upgrade.php');
    global $wpdb;

    self::load_google_fonts();
    self::register_table_names();

    $stats = "CREATE TABLE IF NOT EXISTS " . $wpdb->lf_stats . " (
    id bigint(20) unsigned NOT NULL AUTO_INCREMENT,
    date date NOT NULL,
    object_id int(10) unsigned NOT NULL,
    type varchar(16) NOT NULL,
    views int(8) unsigned NOT NULL DEFAULT '0',
    views_box2 int(8) unsigned NOT NULL DEFAULT '0',
    conversions int(8) unsigned NOT NULL DEFAULT '0',
    PRIMARY KEY (id),
    UNIQUE KEY type_date_id (type, date, object_id)
    ) ENGINE=MyISAM CHARACTER SET utf8 COLLATE utf8_general_ci;";

    dbDelta($stats);

    // WP component is always active
    leadflow_external_api::set_provider('wp');
    leadflow_external_api::verify_api(array());
    leadflow_external_api::get_lists(true);

    // note installation date for stats
    update_option('lf_installation_timestamp', current_time('timestamp'), false);
  } // activate


  static function widgets_init() {
    register_widget('leadflow_widget_optin');
  } // widgets_init


  // clean up on deactivate
  static function deactivate($delete_cache = false) {
    global $wpdb;

    delete_option(LF_POINTERS);
    delete_option(LF_EVENTS);
    delete_option('lf-google-fonts');
    delete_option('lf_flush_rewrite');
    delete_option('lf_jvzoo_last_transaction');
    delete_option('lf_jvzoo_processed_transactions');
    delete_option('lf_stars_bulk_cron');
    $wpdb->query($wpdb->prepare('DELETE FROM ' . $wpdb->options . ' WHERE option_name LIKE %s', array('%transient%lf_fullcontact_%')));

    flush_rewrite_rules();
    leadflow_cron::clear();
  } // deactivate


  // remove all after plugin is deleted
  static function uninstall() {
	global $wpdb;
	
    delete_option(LF_OPTIONS);
    delete_option(LF_POINTERS);
    delete_option(LF_EVENTS);
    delete_option('lf-google-fonts');
    delete_option('lf_flush_rewrite');
    delete_option('lf_installation_timestamp');

    $wpdb->query('DROP TABLE IF EXISTS ' . $wpdb->lf_stats);

    $tmp_posts = get_posts(array('post_type' => 'optin-pages', 'post_status' => 'any', 'numberposts' => -1, 'fields' => 'ids'));
    if ($tmp_posts) {
      foreach ($tmp_posts as $tmp_post) {
        wp_delete_post($tmp_post, true);
      }
    }
    $tmp_posts = get_posts(array('post_type' => 'lead-flows', 'post_status' => 'any', 'numberposts' => -1, 'fields' => 'ids'));
    if ($tmp_posts) {
      foreach ($tmp_posts as $tmp_post) {
        wp_delete_post($tmp_post, true);
      }
    }
    $tmp_posts = get_posts(array('post_type' => 'leadflow-abtests', 'post_status' => 'any', 'numberposts' => -1, 'fields' => 'ids'));
    if ($tmp_posts) {
      foreach ($tmp_posts as $tmp_post) {
        wp_delete_post($tmp_post, true);
      }
    }
    $tmp_posts = get_posts(array('post_type' => 'leadflow-subscribers', 'post_status' => 'any', 'numberposts' => -1, 'fields' => 'ids'));
    if ($tmp_posts) {
      foreach ($tmp_posts as $tmp_post) {
        wp_delete_post($tmp_post, true);
      }
    }
  } // uninstall


  static function woocommerce_init() {
    if (!class_exists('WC_Integration')) {
      return;
    }

    function add_leadsflowpro_integration($methods) {
      $methods[] = 'leadflow_woocommerce_integration';

      return $methods;
    }
    add_filter('woocommerce_integrations', 'add_leadsflowpro_integration');

    function action_links($links) {
      $plugin_links = array(
        '<a href="' . admin_url('admin.php?page=wc-settings&tab=integration&section=leadsflowpro') . '">Settings</a>',
      );

      return array_merge( $plugin_links, $links );
    }

    add_filter( 'plugin_action_links_' . plugin_basename( __FILE__ ), 'action_links' );
  }
} // class leadflow

add_action('init', array('leadflow', 'init'));
add_action('widgets_init', array('leadflow', 'widgets_init'));
add_action('plugins_loaded', array('leadflow', 'woocommerce_init'));
add_action('plugins_loaded', array('leadflow_cron', 'plugins_loaded'));
register_activation_hook(__FILE__, array('leadflow', 'activate'));
register_deactivation_hook(__FILE__, array('leadflow', 'deactivate'));
register_uninstall_hook(__FILE__, array('leadflow', 'uninstall'));


// external functions available to other plugins
function lfp_external_flow($flow_id, $data) {
  $flow = get_post((int) $flow_id);

  if (!$flow || $flow->post_type != 'lead-flows') {
    return new WP_Error('Invalid flow ID.');
  }

  if (leadflow_flow_common::get_flow_type((int) $flow_id) != 'external') {
    return new WP_Error('Invalid flow type.');
  }

  $tmp = leadflow_flow_common::process_external_flow($flow, $data);

  return $tmp;
} // lfp_external_flow


// external functions available to other plugins
function lfp_ecommerce_flow($flow_id, $data) {

  $flow = get_post((int) $flow_id);

  if (!$flow || $flow->post_type != 'lead-flows') {
    return new WP_Error('Invalid flow ID.');
  }

  if (leadflow_flow_common::get_flow_type((int) $flow_id) != 'ecommerce') {
    return new WP_Error('Invalid flow type.');
  }

  $tmp = leadflow_flow_common::process_ecommerce_flow($flow, $data);

  return $tmp;
} // lfp_ecommerce_flow